/*
 * Decompiled with CFR 0.152.
 */
package storage;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.util.LinkedList;
import java.util.List;
import java.util.UUID;
import javassist.ClassPool;
import javassist.CtClass;
import javassist.NotFoundException;
import javassist.bytecode.Descriptor;
import org.apache.commons.io.FileUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import storage.CallbackEvent;
import storage.CallbackHandler;
import storage.StorageException;
import storage.StorageObject;
import storage.utils.Serializer;

public final class StorageItf {
    private static final Logger LOGGER = LogManager.getLogger("es.bsc.compss.Storage");
    private static final String ERROR_HOSTNAME = "ERROR: Cannot find localhost hostname";
    private static final String ERROR_CREATE_WD = "ERROR: Cannot create WD ";
    private static final String ERROR_ERASE_WD = "ERROR: Cannot erase WD";
    private static final String ERROR_CONFIGURATION_NOT_FOUND = "ERROR: Configuration file not found";
    private static final String ERROR_CONFIGURATION_CANNOT_OPEN = "ERROR: Cannot open configuration file";
    private static final String ERROR_NO_PSCO = "ERROR: Cannot find PSCO in master with id=";
    private static final String ERROR_NEW_REPLICA = "ERROR: Cannot create new replica of PSCO with id=";
    private static final String ERROR_NEW_VERSION = "ERROR: Cannot create new version of PSCO with id=";
    private static final String ERROR_DESERIALIZE = "ERROR: Cannot deserialize object with id=";
    private static final String ERROR_SERIALIZE = "ERROR: Cannot serialize object to id=";
    private static final String ERROR_METHOD_NOT_FOUND = "ERROR: ExecuteTask Method not found with descriptor ";
    private static final String ERROR_GET_BY_ID = "ERROR: Cannot find target by id on executeTask";
    private static final String ERROR_REFLECTION = "ERROR: Cannot invoke method by reflection in executeTask";
    private static final String ERROR_RETRIEVE_ID = "ERROR: Cannot retrieve PSCO for executeTask with id ";
    private static final String ERROR_CLASS_NOT_FOUND = "ERROR: Target object class not found";
    private static final String BASE_WORKING_DIR = File.separator + "tmp" + File.separator + "PSCO" + File.separator;
    private static final String MASTER_HOSTNAME;
    private static final String MASTER_WORKING_DIR;
    private static final String ID_EXTENSION = ".ID";
    private static final String PSCO_EXTENSION = ".PSCO";
    private static final LinkedList<String> HOSTNAMES;

    public static void init(String storageConf) throws StorageException {
        LOGGER.info("[LOG] Storage Initialization");
        HOSTNAMES.add(MASTER_HOSTNAME);
        LOGGER.info("[LOG] Configuration received: " + storageConf);
        try {
            BufferedReader br = new BufferedReader(new FileReader(storageConf));
            Object object = null;
            try {
                String line;
                while ((line = br.readLine()) != null) {
                    HOSTNAMES.add(line);
                }
            }
            catch (Throwable line) {
                object = line;
                throw line;
            }
            finally {
                if (br != null) {
                    if (object != null) {
                        try {
                            br.close();
                        }
                        catch (Throwable line) {
                            ((Throwable)object).addSuppressed(line);
                        }
                    } else {
                        br.close();
                    }
                }
            }
        }
        catch (FileNotFoundException e) {
            throw new StorageException(ERROR_CONFIGURATION_NOT_FOUND, e);
        }
        catch (IOException e) {
            throw new StorageException(ERROR_CONFIGURATION_CANNOT_OPEN, e);
        }
        File wd = new File(BASE_WORKING_DIR);
        if (!wd.exists()) {
            try {
                wd.mkdir();
            }
            catch (SecurityException se) {
                throw new StorageException(ERROR_CREATE_WD + BASE_WORKING_DIR, se);
            }
        }
        for (String hostname : HOSTNAMES) {
            LOGGER.debug("[LOG] Hostname: " + hostname);
            String hostPath = BASE_WORKING_DIR + hostname;
            File hostWD = new File(hostPath);
            if (hostWD.exists()) continue;
            try {
                hostWD.mkdir();
            }
            catch (SecurityException se) {
                throw new StorageException(ERROR_CREATE_WD + hostPath, se);
            }
        }
        LOGGER.info("[LOG] Storage Initialization finished");
    }

    public static void finish() throws StorageException {
        LOGGER.info("[LOG] Storage Finish");
        try {
            File wd = new File(BASE_WORKING_DIR);
            if (wd.exists()) {
                FileUtils.deleteDirectory(new File(BASE_WORKING_DIR));
            }
        }
        catch (Exception e) {
            LOGGER.warn("[LOG] ERROR: Cannot erase WD", (Throwable)e);
        }
        LOGGER.info("[LOG] Storage Finish finished");
    }

    public static List<String> getLocations(String id) throws StorageException {
        LOGGER.info("[LOG] Get locations of " + id);
        LinkedList<String> result = new LinkedList<String>();
        for (String hostname : HOSTNAMES) {
            LOGGER.info("[LOG] Checking hostname " + hostname);
            String path = BASE_WORKING_DIR + hostname + File.separator + id + ID_EXTENSION;
            File pscoLocation = new File(path);
            if (!pscoLocation.exists()) continue;
            LOGGER.info("[LOG] Hostname " + hostname + " has id. Adding");
            result.add(hostname);
        }
        return result;
    }

    public static void newReplica(String id, String hostName) throws StorageException {
        File target;
        String targetPath;
        LOGGER.info("NEW REPLICA: " + id + " on host " + hostName);
        File source = new File(MASTER_WORKING_DIR + id + ID_EXTENSION);
        if (source.exists()) {
            targetPath = BASE_WORKING_DIR + hostName + File.separator + id + ID_EXTENSION;
            target = new File(targetPath);
            try {
                Files.copy(source.toPath(), target.toPath(), StandardCopyOption.REPLACE_EXISTING);
            }
            catch (IOException e) {
                throw new StorageException(ERROR_NEW_REPLICA + id, e);
            }
        } else {
            throw new StorageException(ERROR_NO_PSCO + id);
        }
        source = new File(MASTER_WORKING_DIR + id + PSCO_EXTENSION);
        if (source.exists()) {
            targetPath = BASE_WORKING_DIR + hostName + File.separator + id + PSCO_EXTENSION;
            target = new File(targetPath);
            try {
                Files.copy(source.toPath(), target.toPath(), StandardCopyOption.REPLACE_EXISTING);
            }
            catch (IOException e) {
                throw new StorageException(ERROR_NEW_REPLICA + id, e);
            }
        } else {
            throw new StorageException(ERROR_NO_PSCO + id);
        }
    }

    public static String newVersion(String id, boolean preserveSource, String hostName) throws StorageException {
        LOGGER.info("NEW VERSION: " + id + " on host " + hostName);
        String newId = "psco_" + UUID.randomUUID().toString();
        File newIdFile = new File(MASTER_WORKING_DIR + newId + ID_EXTENSION);
        try (BufferedWriter br = new BufferedWriter(new FileWriter(newIdFile));){
            br.write(newId);
            br.flush();
        }
        catch (FileNotFoundException e) {
            throw new StorageException(ERROR_NEW_VERSION + id, e);
        }
        catch (IOException e) {
            throw new StorageException(ERROR_NEW_VERSION + id, e);
        }
        File source = new File(MASTER_WORKING_DIR + id + PSCO_EXTENSION);
        if (source.exists()) {
            String targetPath = MASTER_WORKING_DIR + newId + PSCO_EXTENSION;
            File target = new File(targetPath);
            try {
                Files.copy(source.toPath(), target.toPath(), StandardCopyOption.REPLACE_EXISTING);
            }
            catch (IOException e) {
                throw new StorageException(ERROR_NEW_VERSION + id + " to " + targetPath, e);
            }
        } else {
            throw new StorageException(ERROR_NO_PSCO + id);
        }
        return newId;
    }

    public static Object getByID(String id) throws StorageException {
        for (String hostname : HOSTNAMES) {
            String path = BASE_WORKING_DIR + hostname + File.separator + id + PSCO_EXTENSION;
            File source = new File(path);
            if (!source.exists()) continue;
            try {
                Object obj = Serializer.deserialize(path);
                ((StorageObject)obj).setID(id);
                return obj;
            }
            catch (ClassNotFoundException e) {
                throw new StorageException(ERROR_DESERIALIZE + id, e);
            }
            catch (IOException e) {
                throw new StorageException(ERROR_DESERIALIZE + id, e);
            }
        }
        throw new StorageException(ERROR_NO_PSCO + id);
    }

    public static String executeTask(final String id, final String descriptor, final Object[] values, String hostName, final CallbackHandler callback) throws StorageException {
        LOGGER.info("EXECUTE TASK: " + descriptor + " on host " + hostName);
        String localUUID = UUID.randomUUID().toString();
        new Thread(localUUID){

            @Override
            public void run() {
                try {
                    Object obj = null;
                    try {
                        obj = StorageItf.getByID(id);
                    }
                    catch (StorageException se) {
                        throw new StorageException(StorageItf.ERROR_RETRIEVE_ID + id, se);
                    }
                    if (obj == null) {
                        throw new StorageException(StorageItf.ERROR_RETRIEVE_ID + id);
                    }
                    LOGGER.info("- Target object is PSCO with class " + obj.getClass() + " with id = " + id);
                    ClassPool pool = ClassPool.getDefault();
                    Method methodToExecute = null;
                    for (Method methodAvailable : obj.getClass().getDeclaredMethods()) {
                        String methodAvailableDescriptor;
                        int n = methodAvailable.getParameterAnnotations().length;
                        Class<?>[] cParams = methodAvailable.getParameterTypes();
                        CtClass[] ctParams = new CtClass[n];
                        for (int i = 0; i < n; ++i) {
                            try {
                                ctParams[i] = pool.getCtClass(cParams[i].getName());
                                continue;
                            }
                            catch (NotFoundException e) {
                                throw new Exception("ERROR: Target object class not found " + cParams[i].getName(), e);
                            }
                        }
                        try {
                            methodAvailableDescriptor = methodAvailable.getName() + Descriptor.ofMethod(pool.getCtClass(methodAvailable.getReturnType().getName()), ctParams);
                        }
                        catch (NotFoundException e) {
                            throw new Exception("ERROR: Target object class not found " + methodAvailable.getReturnType().getName(), e);
                        }
                        if (!descriptor.equals(methodAvailableDescriptor)) continue;
                        methodToExecute = methodAvailable;
                        break;
                    }
                    if (methodToExecute == null) {
                        throw new Exception(StorageItf.ERROR_METHOD_NOT_FOUND + descriptor);
                    }
                    Object retValue = methodToExecute.invoke(obj, values);
                    callback.eventListener(new CallbackEvent(this.getName(), CallbackEvent.EventType.SUCCESS, retValue));
                }
                catch (StorageException se) {
                    LOGGER.error(StorageItf.ERROR_GET_BY_ID, (Throwable)se);
                    callback.eventListener(new CallbackEvent(this.getName(), CallbackEvent.EventType.FAIL, se.getMessage()));
                }
                catch (IllegalAccessException | InvocationTargetException e) {
                    LOGGER.error(StorageItf.ERROR_REFLECTION, (Throwable)e);
                    callback.eventListener(new CallbackEvent(this.getName(), CallbackEvent.EventType.FAIL, e.getMessage()));
                }
                catch (Exception e) {
                    LOGGER.error("EXCEPTION ON ExecuteTask", (Throwable)e);
                    callback.eventListener(new CallbackEvent(this.getName(), CallbackEvent.EventType.FAIL, e.getMessage()));
                }
            }
        }.start();
        return localUUID;
    }

    public static Object getResult(CallbackEvent event) throws StorageException {
        LOGGER.info("Get result");
        try {
            return event.getContent();
        }
        catch (Exception e) {
            throw new StorageException(e);
        }
    }

    public static void consolidateVersion(String idFinal) throws StorageException {
        LOGGER.info("Consolidating version for " + idFinal);
    }

    public static void makePersistent(Object o, String id) throws StorageException {
        File idFile = new File(MASTER_WORKING_DIR + id + ID_EXTENSION);
        try (BufferedWriter br = new BufferedWriter(new FileWriter(idFile));){
            br.write(id);
            br.flush();
        }
        catch (FileNotFoundException e) {
            throw new StorageException(ERROR_NEW_VERSION + id, e);
        }
        catch (IOException e) {
            throw new StorageException(ERROR_NEW_VERSION + id, e);
        }
        String path = MASTER_WORKING_DIR + id + PSCO_EXTENSION;
        try {
            Serializer.serialize(o, path);
        }
        catch (IOException e) {
            throw new StorageException(ERROR_SERIALIZE + id, e);
        }
    }

    public static void removeById(String id) {
        for (String hostname : HOSTNAMES) {
            String pscoPath;
            File pscoSource;
            String idPath = BASE_WORKING_DIR + hostname + File.separator + id + ID_EXTENSION;
            File idSource = new File(idPath);
            if (idSource.exists()) {
                idSource.delete();
            }
            if (!(pscoSource = new File(pscoPath = BASE_WORKING_DIR + hostname + File.separator + id + PSCO_EXTENSION)).exists()) continue;
            pscoSource.delete();
        }
    }

    static {
        HOSTNAMES = new LinkedList();
        String hostname = null;
        try {
            InetAddress localHost = InetAddress.getLocalHost();
            hostname = localHost.getCanonicalHostName();
        }
        catch (UnknownHostException e) {
            System.err.println(ERROR_HOSTNAME);
            e.printStackTrace();
            System.exit(1);
        }
        MASTER_HOSTNAME = hostname;
        MASTER_WORKING_DIR = BASE_WORKING_DIR + File.separator + MASTER_HOSTNAME + File.separator;
    }
}

