/*
 * Decompiled with CFR 0.152.
 */
package es.bsc.conn.slurm;

import es.bsc.conn.Connector;
import es.bsc.conn.clients.exceptions.ConnClientException;
import es.bsc.conn.clients.slurm.JobDescription;
import es.bsc.conn.clients.slurm.SlurmClient;
import es.bsc.conn.exceptions.ConnException;
import es.bsc.conn.types.HardwareDescription;
import es.bsc.conn.types.InstallationDescription;
import es.bsc.conn.types.Processor;
import es.bsc.conn.types.SoftwareDescription;
import es.bsc.conn.types.StarterCommand;
import es.bsc.conn.types.VirtualResource;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

public class SlurmConnector
extends Connector {
    private static final String RUNNING = "RUNNING";
    private static final String FAILED = "FAILED";
    private static final long POLLING_INTERVAL = 5L;
    private static final int TIMEOUT = 1800;
    private static final String JOB_STATE = "JobState";
    private static final Logger LOGGER = LogManager.getLogger("es.bsc.compss.Connectors.Conn.SLURM");
    private static final boolean DEBUG = LOGGER.isDebugEnabled();
    private final Map<String, HardwareDescription> vmidToHardwareRequest = new HashMap<String, HardwareDescription>();
    private final Map<String, SoftwareDescription> vmidToSoftwareRequest = new HashMap<String, SoftwareDescription>();
    private final Map<String, String> vmidToHostName = new HashMap<String, String>();
    private final String logDir;
    private final SlurmClient client;
    private final String network;
    private final boolean memInRequest;
    private int currentNodes;

    public SlurmConnector(Map<String, String> props) throws ConnException {
        super(props);
        String appLogdir = System.getProperty("compss.appLogDir");
        if (appLogdir == null) {
            throw new ConnException("[Connector] Unable to get app log dir");
        }
        File f = new File(appLogdir + File.separator + "slurm-conn-log");
        if (!f.mkdirs()) {
            throw new ConnException("[Connector] Unable to create SLURM connector log dir");
        }
        this.logDir = f.getAbsolutePath();
        String masterName = props.get("master_name");
        if (masterName == null || masterName.isEmpty()) {
            throw new ConnException("[Connector] Unable to get master_name. Property is empty");
        }
        boolean ssh = false;
        String sshStr = props.get("slurm_over_ssh");
        if (sshStr != null && !sshStr.isEmpty()) {
            ssh = Boolean.parseBoolean(sshStr);
        }
        boolean expand = true;
        String expandStr = props.get("expand_job");
        if (expandStr != null && !expandStr.isEmpty()) {
            expand = Boolean.parseBoolean(expandStr);
        }
        LOGGER.debug("[Connector] Starting Slurm client (jobName) for master in " + masterName + ", ssh " + ssh + " and expand " + expand);
        this.client = new SlurmClient(masterName, ssh, expand);
        String networkProp = props.get("network");
        this.network = networkProp == null ? "" : networkProp;
        String memInReqStr = props.get("mem_in_req");
        this.memInRequest = memInReqStr != null && !memInReqStr.isEmpty() ? Boolean.parseBoolean(memInReqStr) : false;
        this.currentNodes = 0;
    }

    @Override
    public Object create(String requestName, HardwareDescription hd, SoftwareDescription sd, Map<String, String> prop, StarterCommand starterCMD) throws ConnException {
        try {
            String jobName = requestName;
            ++this.currentNodes;
            String jobId = this.client.createCompute(this.generateJobDescription(hd, sd), this.generateExecScript(jobName, hd, sd, prop, starterCMD));
            this.vmidToHardwareRequest.put(jobId, hd);
            this.vmidToSoftwareRequest.put(jobId, sd);
            VirtualResource vr = new VirtualResource(jobId, hd, sd, prop);
            return vr.getId();
        }
        catch (ConnClientException ce) {
            LOGGER.error("[Connector] Exception submitting job for creation", (Throwable)ce);
            --this.currentNodes;
            throw new ConnException(ce);
        }
        catch (Exception e) {
            LOGGER.error("[Connector] Exception submitting job for node creation", (Throwable)e);
            throw new ConnException(e);
        }
    }

    @Override
    public Object[] createMultiple(int replicas, String requestName, HardwareDescription hd, SoftwareDescription sd, Map<String, String> prop, StarterCommand starterCMD) throws ConnException {
        Object[] envIds = new Object[replicas];
        for (int i = 0; i < replicas; ++i) {
            envIds[i] = this.create(requestName, hd, sd, prop, starterCMD);
        }
        return envIds;
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private String generateExecScript(String jobName, HardwareDescription hd, SoftwareDescription sd, Map<String, String> prop, StarterCommand starterCMD) throws ConnException {
        String stdFlags = this.generateFlags(jobName, prop);
        InstallationDescription instDesc = sd.getInstallation();
        StringBuilder script = new StringBuilder("#!/bin/sh\n");
        String launchCommand = prop.get("launch_command");
        if (launchCommand != null && !launchCommand.isEmpty()) {
            LOGGER.debug("[Connector] Adding launch command to the run script");
            script.append(launchCommand + "$SLURM_JOB_NODELIST ");
        } else {
            LOGGER.debug("[Connector] launch command not found");
        }
        String installDir = instDesc.getInstallDir();
        if (installDir == null && (installDir = System.getenv("COMPSS_HOME")) == null) {
            throw new ConnException("[Connector] Unable to get COMPSs installation directory");
        }
        if (hd.getImageName() != null && !hd.getImageName().isEmpty() && !"None".equals(hd.getImageName())) {
            script.append("singularity exec ");
            String containerOpts = prop.get("container_opts");
            if (containerOpts != null && !containerOpts.isEmpty()) {
                script.append(containerOpts + " ");
            }
            script.append(hd.getImageName());
        }
        starterCMD.setScriptName(installDir + "/Runtime/scripts/system/adaptors/nio/persistent_worker_starter.sh");
        starterCMD.setWorkerName(" $SLURM_JOB_NODELIST" + this.network);
        starterCMD.setNodeId(String.valueOf(this.client.getInitialNodes() + this.currentNodes));
        try {
            String[] cmd;
            for (String s : cmd = starterCMD.getStartCommand()) {
                script.append(" " + s);
            }
        }
        catch (Exception e) {
            throw new ConnException(e);
        }
        File runScript = new File(this.logDir + File.separator + "run_" + jobName);
        try {
            if (!runScript.createNewFile()) {
                throw new IOException("[Connector] ERROR: File already exists");
            }
            if (!runScript.setExecutable(true)) {
                throw new IOException("[Connector] ERROR: Cannot make the file executable");
            }
        }
        catch (IOException ioe) {
            throw new ConnException("[Connector] Exception creating script", ioe);
        }
        try (FileOutputStream fos = new FileOutputStream(runScript);){
            fos.write(script.toString().getBytes());
            String string = stdFlags + " " + runScript.getAbsolutePath();
            return string;
        }
        catch (IOException e) {
            throw new ConnException("[Connector] Exception writting script", e);
        }
    }

    private String generateFlags(String jobName, Map<String, String> prop) {
        String constraint;
        String qos;
        String reservation;
        StringBuilder flags = new StringBuilder("--job-name=" + jobName + " -e " + this.logDir + File.separator + jobName + ".err -o " + this.logDir + File.separator + jobName + ".out");
        String queue = prop.get("queue");
        if (queue != null && !queue.isEmpty()) {
            flags.append(" -p " + queue);
        }
        if ((reservation = prop.get("reservation")) != null && !reservation.isEmpty()) {
            flags.append(" --reservation=" + reservation);
        }
        if ((qos = prop.get("qos")) != null && !qos.isEmpty()) {
            flags.append(" --qos=" + qos);
        }
        if ((constraint = prop.get("constraint")) != null && !constraint.isEmpty()) {
            flags.append(" --constraint=" + constraint);
        }
        return flags.toString();
    }

    private JobDescription generateJobDescription(HardwareDescription hd, SoftwareDescription sd) {
        if (DEBUG) {
            LOGGER.debug("Generating Job description from:");
            LOGGER.debug(" - HD: " + hd);
            LOGGER.debug(" - SD: " + sd);
        }
        HashMap<String, String> req = new HashMap<String, String>();
        req.put("NumNodes", "1");
        req.put("NumCPUs", Integer.toString(hd.getTotalCPUComputingUnits()));
        if (this.memInRequest && hd.getMemorySize() > 0.0f) {
            req.put("mem", Integer.toString((int)hd.getMemorySize() * 1024));
        }
        int gpuUnits = 0;
        for (Processor p : hd.getProcessors()) {
            if (!"GPU".equals(p.getArchitecture())) continue;
            gpuUnits += p.getComputingUnits();
        }
        if (gpuUnits > 0) {
            req.put("Gres", Integer.toString(gpuUnits));
        } else if (hd.getTotalGPUComputingUnits() > 0) {
            req.put("Gres", "gpu:" + Integer.toString(hd.getTotalGPUComputingUnits()));
        }
        return new JobDescription(req);
    }

    @Override
    public VirtualResource waitUntilCreation(Object id) throws ConnException {
        LOGGER.debug("[Connector] Waiting for resource creation " + id);
        String jobId = (String)id;
        LOGGER.debug("[Connector] Waiting until node of job " + jobId + " is created");
        try {
            JobDescription jd = this.client.getJobDescription(jobId);
            LOGGER.debug("[Connector] Job State is " + jd.getProperty(JOB_STATE));
            int tries = 0;
            while (jd.getProperty(JOB_STATE) == null || !jd.getProperty(JOB_STATE).equals(RUNNING)) {
                if (jd.getProperty(JOB_STATE).equals(FAILED)) {
                    LOGGER.error("[Connector] Error waiting for VM Creation. Middleware has return an error state");
                    throw new ConnException("[Connector] Error waiting for VM Creation. Middleware has return an error state");
                }
                if ((long)tries * 5L > 1800L) {
                    this.client.cancelJob(jobId);
                    this.vmidToHardwareRequest.remove(jobId);
                    this.vmidToSoftwareRequest.remove(jobId);
                    throw new ConnException("[Connector] Maximum Job creation time reached.");
                }
                ++tries;
                Thread.sleep(5000L);
                jd = this.client.getJobDescription(jobId);
                LOGGER.debug("[Connector] Job State is " + jd.getProperty(JOB_STATE));
            }
            this.client.addNodesToMain(jobId, jd);
            VirtualResource vr = new VirtualResource();
            vr.setId(jobId);
            String resourceName = jd.getNodeList().get(0) + this.network;
            LOGGER.debug("[Connector] Setting resource ip: " + resourceName);
            this.vmidToHostName.put(jobId, resourceName);
            vr.setIp(resourceName);
            vr.setProperties(null);
            HardwareDescription hd = this.vmidToHardwareRequest.get(jobId);
            if (hd == null) {
                throw new ConnException("[Connector] Unregistered hardware description for job " + jobId);
            }
            vr.setHd(hd);
            SoftwareDescription sd = this.vmidToSoftwareRequest.get(jobId);
            if (sd == null) {
                throw new ConnException("[Connector] Unregistered software description for job " + jobId);
            }
            sd.setOperatingSystemType("Linux");
            vr.setSd(sd);
            return vr;
        }
        catch (ConnClientException | InterruptedException e) {
            LOGGER.error("[Connector] Exception waiting for VM Creation");
            throw new ConnException(e);
        }
    }

    @Override
    public void destroy(Object id) {
        String jobId = (String)id;
        LOGGER.debug("[Connector] Destroying node for job " + jobId);
        try {
            String resourceName = this.vmidToHostName.get(jobId);
            if (resourceName != null && !resourceName.isEmpty()) {
                if (!this.network.isEmpty()) {
                    resourceName = resourceName.substring(0, resourceName.indexOf(this.network));
                }
                LOGGER.debug("[Connector] Deleting node " + resourceName);
                this.client.deleteCompute(resourceName);
            } else {
                LOGGER.debug("[Connector] Node not found cancelling job " + jobId);
            }
            this.vmidToHardwareRequest.remove(jobId);
            this.vmidToSoftwareRequest.remove(jobId);
            this.vmidToHostName.remove(jobId);
        }
        catch (ConnClientException cce) {
            LOGGER.error("[Connector] Exception waiting for Node Destruction", (Throwable)cce);
        }
        LOGGER.debug("[Connector] Node for job " + jobId + " destroyed.");
        --this.currentNodes;
    }

    @Override
    public float getPriceSlot(VirtualResource virtualResource) {
        return virtualResource.getHd().getPricePerUnit();
    }

    @Override
    public void close() {
    }
}

