#!/usr/bin/python

# -*- coding: utf-8 -*-

# For better print formatting
from __future__ import print_function

# Imports
import unittest
import logging

#
# Logger definition
#

logger = logging.getLogger(__name__)


#
# Translator class
#

class Py2Scop(object):
    """
    Creates an object to translate a given python function into OpenScop

    Attributes:
            - func : Python function object to replace
            - func_code : Full code of func
            - func_ast : AST representation of func
            - for_blocks : List of AST blocks that represent main fors
            - scops: List of OpenScop code that represent main fors
    """

    # TODO: 2 consecutive for's as single for block

    def __init__(self, func=None):
        """
        Initializes the translator internal structures

        Arguments:
                :function func: Python function to translate
        Return:
        Raise:
                - Py2ScopException
        """

        if __debug__:
            logger.debug("[py2scop] Creating translator for " + str(func))

        # Retrieve function to translate
        self.func = func

        # Create AST
        import inspect
        import ast
        try:
            self.func_code = inspect.getsource(func)
            self.func_ast = ast.parse(self.func_code)
        except Exception as e:
            raise Py2ScopException("ERROR: Cannot retrieve AST from function", e)

        # Initialize other variables
        self.for_blocks = None
        self.scops = None

    # WRITE OS TO FILE
    @staticmethod
    def write_os(scop, file_name):
        """
        Writes the given SCOP into the given OpenScop file

        Arguments:
                - scop : SCOP object
                - file_name : Output file path
        Return:
        Raise:
        """

        with open(file_name, 'w') as f:
            print("# [File generated by the OpenScop Library 0.9.1]", file=f)
            print("", file=f)
            scop.write_os(f)

    # TRANSLATE
    def translate(self, base_file_name):
        """
        Inputs a Python code with scop pragmas and outputs its
        OpenScop representation in the given file

        Arguments:
                - base_file_name : OpenScop base name for output file path
        Return:
                - output_file_names : List of written OS files (one per loop block)
        Raise:
                - Py2ScopException
        """

        if __debug__:
            logger.debug("[py2scop] Begin OpenScop translation")

        # Generate the list of for blocks
        if __debug__:
            logger.debug("[py2scop] Extracting for blocks")
        try:
            self.for_blocks = Py2Scop._ast_extract_for_blocks(self.func_ast)
        except Exception as e:
            raise Py2ScopException("ERROR: Cannot generate code blocks", e)
        if __debug__:
            logger.debug("[py2scop] Found " + str(len(self.for_blocks)) + " blocks")

        # Translate loop blocks
        if __debug__:
            logger.debug("[py2scop] Translating for blocks")
        self.scops = []
        try:
            if self.for_blocks is not None:
                for fb_index, fb in enumerate(self.for_blocks):
                    if __debug__:
                        logger.debug("[py2scop] Translating " + str(fb))
                        # import ast
                        # logger.debug(ast.dump(fb))
                    self.scops.append(Py2Scop._ast2scop(fb, fb_index))
        except Exception as e:
            raise Py2ScopException("ERROR: Cannot generate SCOPs from ForBlocks", e)

        # Write loop blocks
        if __debug__:
            logger.debug("[py2scop] Writing each OS to file")
        output_file_names = []
        num_files = 0
        for scop in self.scops:
            file_name = base_file_name + str(num_files)
            try:
                Py2Scop.write_os(scop, file_name)
                if __debug__:
                    logger.debug("[py2scop] Scop written to " + str(file_name))
            except Exception as e:
                raise Py2ScopException("ERROR: Cannot write OS file " + str(file_name), e)

            num_files = num_files + 1
            output_file_names.append(file_name)

        # Add a warn
        if num_files == 0:
            logger.error("WARN: No for loop found. No SCOP file generated")

        if __debug__:
            logger.debug("[py2scop] Translation done")

        # Return written fileNames
        return output_file_names

    # Process AST code
    @staticmethod
    def _ast_extract_for_blocks(node, for_level=0, for_blocks=None):
        """
        Inputs an AST node and process it and its children recursively

        Arguments:
                - node : AST node
                - for_level : Nested FOR level
                - for_blocks : Primary for_blocks found recursively
        Return:
                - for_blocks : Primary for_blocks
        Raise:
        """

        import ast
        import copy

        # Copy current node if it is an outermost loop
        if isinstance(node, ast.For) and for_level == 0:
            node_copy = copy.deepcopy(node)
            ast.fix_missing_locations(node_copy)
            if for_blocks is None:
                for_blocks = []
            for_blocks.append(node_copy)

        # Prepare for next recursion
        if isinstance(node, ast.For):
            for_level = for_level + 1

        # Child recursion
        for _, value in ast.iter_fields(node):
            if isinstance(value, list):
                for item in value:
                    if isinstance(item, ast.AST):
                        for_blocks = Py2Scop._ast_extract_for_blocks(item, for_level, for_blocks)
            elif isinstance(value, ast.AST):
                for_blocks = Py2Scop._ast_extract_for_blocks(value, for_level, for_blocks)

        # Return all the outermost loops
        return for_blocks

    # TRANSLATION FROM AST TO SCOP
    @staticmethod
    def _ast2scop(tree, tree_id):
        """
        Inputs a AST loop block representation and returns its SCOP
        representation

        Arguments:
                - tree : AST loop block representation
                - tree_id : Main for index
        Return:
                - scop : SCOP object representing the loop block
        Raise:
        """

        # Generate global
        from pycompss.util.translators.scop_types.scop.global_class import Global
        from pycompss.util.translators.scop_types.scop.globl.context_class import Context
        from pycompss.util.translators.scop_types.scop.globl.context_class import ContextType
        from pycompss.util.translators.scop_types.scop.globl.parameters_class import Parameters
        from pycompss.util.translators.scop_types.scop.globl.parameters.parameter_class import Parameter
        # Iteration variables
        iter_vars_list = Py2Scop._get_iter_vars(tree)
        iter_vars_set = set(iter_vars_list)  # Erase repeated
        # Array variables
        array_vars_list = Py2Scop._get_array_vars(tree)
        array_vars_set = set(array_vars_list)  # Erase repeated
        # All variables
        all_vars_list = Py2Scop._get_all_params(tree)
        all_vars_set = set(all_vars_list)  # Erase repeated
        all_vars_list = list(all_vars_set)
        # Param variables
        param_vars_set = all_vars_set - iter_vars_set - array_vars_set
        param_vars_list = list(param_vars_set)

        lang = "p"
        num_cols = 2 + len(param_vars_list)
        context_scop = Context(ContextType.CONTEXT, 0, num_cols, 0, 0, 0, len(param_vars_list))
        params_scop = Parameters([Parameter("strings", param_vars_list)])
        global_scop = Global(lang, context_scop, params_scop)

        # Generate statements
        statements_scop = Py2Scop._get_statements(tree, None, [tree_id], param_vars_list, all_vars_list)

        # Generate extensions
        from pycompss.util.translators.scop_types.scop.extensions_class import Extensions
        # from pycompss.util.translators.scop_types.scop.extensions.scatnames_class import Scatnames
        from pycompss.util.translators.scop_types.scop.extensions.arrays_class import Arrays
        from pycompss.util.translators.scop_types.scop.extensions.coordinates_class import Coordinates
        scatnames_scop = None
        arrays_scop = Arrays(all_vars_list)
        coordinates_scop = Coordinates("_autogen.py", 0, 0, 0, 0, 4)
        extensions_scop = Extensions(scatnames_scop, arrays_scop, coordinates_scop)

        # Generate SCOP
        from pycompss.util.translators.scop_types.scop_class import Scop
        scop = Scop(global_scop, statements_scop, extensions_scop)

        return scop

    @staticmethod
    def _get_all_params(node):
        """
        Returns the all the global parameters used within the AST tree

        Arguments:
                - node : head node
        Return:
                - global_params : list of global parameters
        Raise:
        """

        import ast

        # Process current node
        if isinstance(node, ast.Name):
            return [node.id]

        # Child recursion
        global_params = []
        for field, value in ast.iter_fields(node):
            if field == "func" or field == "keywords":
                pass
            else:
                # Any other construction
                if isinstance(value, list):
                    for item in value:
                        if isinstance(item, ast.AST):
                            global_params.extend(Py2Scop._get_all_params(item))
                elif isinstance(value, ast.AST):
                    global_params.extend(Py2Scop._get_all_params(value))

        # Return
        return global_params

    @staticmethod
    def _get_iter_vars(node):
        """
        Returns the all the iteration variables used within the AST tree

        Arguments:
                - node : head node
        Return:
                - iter_vars : list of global iteration variables
        Raise:
        """

        import ast

        # Process current node
        iter_vars = []
        if isinstance(node, ast.For):
            iter_vars.append(node.target.id)

        # Child recursion
        for _, value in ast.iter_fields(node):
            if isinstance(value, list):
                for item in value:
                    if isinstance(item, ast.AST):
                        iter_vars.extend(Py2Scop._get_iter_vars(item))
            elif isinstance(value, ast.AST):
                iter_vars.extend(Py2Scop._get_iter_vars(value))

        # Return
        return iter_vars

    @staticmethod
    def _get_array_vars(node):
        """
        Returns the all the array names used within the AST tree

        Arguments:
                - node : head node
        Return:
                - array_vars : list of array names
        Raise:
        """

        import ast

        # Process current node
        array_vars = []
        if isinstance(node, ast.Subscript):
            if isinstance(node.value, ast.Name):
                array_vars.append(node.value.id)

        # Child recursion
        for _, value in ast.iter_fields(node):
            if isinstance(value, list):
                for item in value:
                    if isinstance(item, ast.AST):
                        array_vars.extend(Py2Scop._get_array_vars(item))
            elif isinstance(value, ast.AST):
                array_vars.extend(Py2Scop._get_array_vars(value))

        # Return
        return array_vars

    @staticmethod
    def _get_statements(node, for_fathers, scatter_indexes, param_vars, all_vars):
        """
        Returns the processed SCOP statements found inside the AST tree

        Arguments:
                - node : head node
                - for_fathers : list of for loops on top of the statement
                - scatter_indexes : list of scatter indexes for the statement
                - param_vars : List of parameter variables
                - all_vars : List of parameter, iteration and access variables
        Return:
                - statements_scop : List of processed SCOP statements
        Raise:
        """

        import ast

        # Process current node
        if isinstance(node, ast.Assign) or isinstance(node, ast.AugAssign) or isinstance(node, ast.Expr):
            s_scop = Py2Scop._process_statement(node, for_fathers, scatter_indexes, param_vars, all_vars)
            return [s_scop]

        # Insert for to fathers' list
        if isinstance(node, ast.For):
            if for_fathers is None:
                for_fathers = []
            for_fathers.append(node)

        # Add scatter depth index (top level is always a FOR so we always increase when processing children)
        scatter_indexes.append(0)

        # Children recursion
        statements_scop = []
        child_index = 0
        for _, value in ast.iter_fields(node):
            if isinstance(value, list):
                for item in value:
                    if isinstance(item, ast.AST):
                        scatter_indexes[-1] = child_index
                        statements_scop.extend(Py2Scop._get_statements(item, for_fathers, scatter_indexes,
                                                                       param_vars, all_vars))
                        # Only increase child index if it a special node
                        if isinstance(item, ast.For) or isinstance(item, ast.Assign) \
                                or isinstance(item, ast.AugAssign) or isinstance(item, ast.Expr):
                            child_index += 1
            elif isinstance(value, ast.AST):
                scatter_indexes[-1] = child_index
                statements_scop.extend(Py2Scop._get_statements(value, for_fathers, scatter_indexes, param_vars,
                                                               all_vars))
                # Only increase child index if it a special node
                if isinstance(value, ast.For) or isinstance(value, ast.Assign) \
                        or isinstance(value, ast.AugAssign) or isinstance(value, ast.Expr):
                    print("INCREASE2")
                    child_index += 1

        # Remove scatter depth index
        scatter_indexes.pop()

        # Remove from fathers
        if isinstance(node, ast.For):
            for_fathers.pop()

        # Return all scop processed statements
        return statements_scop

    @staticmethod
    def _process_statement(statement_loop, fathers_loop, scatter_indexes, param_vars, all_vars):
        """
        Process the current AST node to generate its SCOP representation

        Arguments:
                - statement_loop : AST node representing the statement
                - fathers_loop : List of AST nodes representing the loops surrounding the statement
                - scatter_indexes : list of scatter indexes for the statement
                - param_vars : List of parameter variables
                - all_vars : List of parameter, iteration, and access variables
        Return:
                - s_scop : SCOP representation of the AST node
        Raise:
                - Py2ScopException
        """

        import ast

        from pycompss.util.translators.scop_types.scop.statement_class import Statement
        from pycompss.util.translators.scop_types.scop.statement.relation_class import Relation
        from pycompss.util.translators.scop_types.scop.statement.relation_class import RelationType
        from pycompss.util.translators.scop_types.scop.statement.statement_extension_class import StatementExtension

        # Retrieve iteration variables
        iter_vars = []
        for f in fathers_loop:
            iter_vars.append(f.target.id)

        # Retrieve control variables for loop position, loop increment and statement
        control_vars = []
        num_control_vars = 2 * len(fathers_loop) + 1
        for c_index in range(1, num_control_vars + 1):
            cv = "c" + str(c_index)
            control_vars.append(cv)

        # DOMAIN
        domain_rows = 2 * len(fathers_loop)  # 2 constraints per loop
        domain_cols = 1 + len(iter_vars) + len(param_vars) + 1
        domain_out_dims = len(iter_vars)
        domain_in_dims = 0
        domain_local_dims = 0
        domain_num_pars = len(param_vars)
        domain_matrix = [[0 for _ in range(domain_cols)] for _ in range(domain_rows)]
        # Create dictionary from col name to col index
        names2index = {'e/i': 0}
        index = 1
        for iv in iter_vars:
            names2index[iv] = index
            index = index + 1
        for pv in param_vars:
            names2index[pv] = index
            index = index + 1
        names2index['indep'] = index
        # Add domain loop bounds
        row_id = 0
        for f in fathers_loop:
            loop_var = f.target.id

            # LOWER BOUND
            if len(f.iter.args) == 1:
                # For loop expression is of the form range(N), we don't process expr
                ub_index = 0
            else:
                # For loop expression is of the form range(N,M), we process expr
                ub_index = 1
                # We mark the expression
                domain_matrix[row_id] = Py2Scop._process_expr(f.iter.args[0], names2index, domain_cols)
                # i >= expr --> i - expr >= 0
                domain_matrix[row_id] = [-x for x in domain_matrix[row_id]]
            #  We mark it as an inequality
            domain_matrix[row_id][names2index['e/i']] = 1
            # We mark the loop variable
            domain_matrix[row_id][names2index[loop_var]] = 1
            # Prepare for next row
            row_id = row_id + 1

            # UPPER BOUND
            # We mark the expression
            domain_matrix[row_id] = Py2Scop._process_expr(f.iter.args[ub_index], names2index, domain_cols)
            # i < expr --> i - expr < 0 --> -i + expr > 0 --> -i + expr -1 >= 0
            domain_matrix[row_id][-1] = domain_matrix[row_id][-1] - 1
            # We mark it as an inequality
            domain_matrix[row_id][names2index['e/i']] = 1
            # We mark the loop variable
            domain_matrix[row_id][names2index[loop_var]] = -1
            # Prepare for next row
            row_id = row_id + 1

        domain_scop = Relation(RelationType.DOMAIN, domain_rows, domain_cols, domain_out_dims, domain_in_dims,
                               domain_local_dims, domain_num_pars, domain_matrix)

        # Scattering
        scattering_rows = len(control_vars)
        scattering_cols = 1 + len(control_vars) + len(iter_vars) + len(param_vars) + 1
        scattering_out_dims = len(control_vars)
        scattering_in_dims = len(iter_vars)
        scattering_local_dims = 0
        scattering_num_pars = len(param_vars)
        scattering_matrix = [[0 for _ in range(scattering_cols)] for _ in range(scattering_rows)]
        # Create dictionary from col name to col index
        names2index = {'e/i': 0}
        index = 1
        for cv in control_vars:
            names2index[cv] = index
            index = index + 1
        for iv in iter_vars:
            names2index[iv] = index
            index = index + 1
        for pv in param_vars:
            names2index[pv] = index
            index = index + 1
        names2index['indep'] = index
        # Add scattering values
        for row_ind in range(0, scattering_rows):
            # Mark control variable
            col_ind = row_ind + 1
            scattering_matrix[row_ind][col_ind] = -1

            if row_ind % 2 == 0:
                # Mark scatter index variable
                col_ind = names2index['indep']
                # Maybe this is not a full depth statement
                scatter_index_depth = row_ind / 2
                if scatter_index_depth < len(scatter_indexes):
                    scatter_value = scatter_indexes[scatter_index_depth]
                else:
                    scatter_value = 0
                scattering_matrix[row_ind][col_ind] = scatter_value
            else:
                # Mark iteration variable
                col_ind = 1 + len(control_vars) + row_ind / 2
                scattering_matrix[row_ind][col_ind] = 1

        scattering_scop = Relation(RelationType.SCATTERING, scattering_rows, scattering_cols,
                                   scattering_out_dims, scattering_in_dims, scattering_local_dims,
                                   scattering_num_pars, scattering_matrix)

        # Accesses
        accesses_scop = []
        # Add write access
        if isinstance(statement_loop, ast.Assign):
            # Expr is of the form X1,...,Xn = expr
            if isinstance(statement_loop.targets[0], ast.Subscript):
                # Single return value
                accesses_scop.append(Py2Scop._process_access(statement_loop.targets[0], RelationType.WRITE, iter_vars,
                                                             param_vars, all_vars))
            elif isinstance(statement_loop.targets[0], ast.Tuple):
                # Multiple return value
                for ret_expr in statement_loop.targets[0].elts:
                    accesses_scop.append(Py2Scop._process_access(ret_expr, RelationType.WRITE, iter_vars,
                                                                 param_vars, all_vars))
        elif isinstance(statement_loop, ast.AugAssign):
            # Expr is of the form x += expr
            if isinstance(statement_loop.target, ast.Subscript):
                accesses_scop.append(Py2Scop._process_access(statement_loop.target, RelationType.WRITE, iter_vars,
                                                             param_vars, all_vars))
                accesses_scop.append(Py2Scop._process_access(statement_loop.target, RelationType.READ, iter_vars,
                                                             param_vars, all_vars))
        elif isinstance(statement_loop, ast.Expr):
            # Expr is of the form func(). Only READ access so no need to add anything
            pass

        # Add read accesses
        accesses_scop.extend(Py2Scop._process_accesses(statement_loop.value, iter_vars, param_vars, all_vars))

        # Extensions
        import astor
        from pycompss.util.translators.astor_source_gen.pycompss_source_gen import PyCOMPSsSourceGen
        e_scop = StatementExtension(iter_vars,
                                    astor.to_source(statement_loop,
                                                    pretty_source=PyCOMPSsSourceGen.long_long_line_ps))
        extensions_scop = [e_scop]

        # Create and return statement
        s_scop = Statement(domain_scop, scattering_scop, accesses_scop, extensions_scop)
        return s_scop

    @staticmethod
    def _process_accesses(node, iter_vars, param_vars, all_vars):
        """
        Processes the accesses to array variables from the given AST node

        Arguments:
                - node : Main access AST node
                - iter_vars : List of iteration variables
                - param_vars : List of parameter variables
                - all_vars : List of parameter, iteration, and access variables
        Return:
                - accesses : List of SCOP Relation accesses
        Raise:
        """

        import ast

        accesses = []

        if isinstance(node, ast.Name):
            # Nothing to do since we control array accesses
            pass
        elif isinstance(node, ast.Num):
            # Nothing to do since we control array accesses
            pass
        elif isinstance(node, ast.Subscript):
            # Process array access
            from pycompss.util.translators.scop_types.scop.statement.relation_class import RelationType
            accesses.append(Py2Scop._process_access(node, RelationType.READ, iter_vars,
                                                    param_vars, all_vars))
        elif isinstance(node, ast.BinOp):
            accesses.extend(Py2Scop._process_accesses(node.left, iter_vars, param_vars, all_vars))
            accesses.extend(Py2Scop._process_accesses(node.right, iter_vars, param_vars, all_vars))
        elif isinstance(node, ast.Call):
            for call_arg in node.args:
                accesses.extend(Py2Scop._process_accesses(call_arg, iter_vars, param_vars, all_vars))

        return accesses

    @staticmethod
    def _process_access(node, access_type, iter_vars, param_vars, all_vars):
        """
        Process the current array access to SCOP format

        Arguments:
                - node : Access AST node
                - access_type : SCOP access mode
                - iter_vars : List of iteration variables
                - param_vars : List of parameter variables
                - all_vars : List of parameter, iteration, and access variables
        Return:
                - access : Scop Relation access object
        Raise:
        """

        import ast

        # Obtain array dimensions
        array_dims = 1
        n2 = node
        while isinstance(n2, ast.Subscript):
            n2 = n2.value
            array_dims = array_dims + 1
        accessed_var = n2.id

        # Access
        access_rows = array_dims
        access_cols = 1 + array_dims + len(iter_vars) + len(param_vars) + 1
        access_out_dims = array_dims
        access_in_dims = len(iter_vars)
        access_local_dims = 0
        access_num_pars = len(param_vars)
        access_matrix = [[0 for _ in range(access_cols)] for _ in range(access_rows)]

        # Create dictionary from col name to col index
        names2index = {'e/i': 0}
        index = 1
        for i in range(array_dims):
            ac = "[" + str(i) + "]"
            names2index[ac] = i
            index = index + 1
        for iv in iter_vars:
            names2index[iv] = index
            index = index + 1
        for pv in param_vars:
            names2index[pv] = index
            index = index + 1
        names2index['indep'] = index

        # Fill Access matrix
        # Add base matrix access id
        access_matrix[0][1] = -1
        access_matrix[0][-1] = all_vars.index(accessed_var) + 1
        # Add accesses to subscripts
        col_index = 2
        row_index = 1
        n2 = node
        while isinstance(n2, ast.Subscript):
            # Process access expression
            access_row = Py2Scop._process_expr(n2.slice.value, names2index, access_cols)
            access_matrix[row_index] = access_row
            # Mark accessed subscript
            access_matrix[row_index][col_index] = -1
            # Next iteration
            row_index = row_index + 1
            col_index = col_index + 1
            n2 = n2.value

        # Instantiate Relation and return
        from pycompss.util.translators.scop_types.scop.statement.relation_class import Relation
        access = Relation(access_type, access_rows, access_cols, access_out_dims, access_in_dims,
                          access_local_dims, access_num_pars, access_matrix)
        return access

    @staticmethod
    def _process_expr(node, names2index, dim):
        """
        Converts an expression to its list of assignments

        Arguments:
                - node : Expression AST node
                - names2index : Map from constant names to its indexes
                - dim : Row dimension
        Return:
                - res : list of assignments
        Raise:
                - Py2ScopException
        """

        import ast

        res = [0] * dim
        if isinstance(node, ast.Name):
            res[names2index[node.id]] = 1
        elif isinstance(node, ast.Num):
            res[names2index['indep']] = node.n
        elif isinstance(node, ast.BinOp):
            # Process recursive expression
            res_left = Py2Scop._process_expr(node.left, names2index, dim)
            res_right = Py2Scop._process_expr(node.right, names2index, dim)
            # Merge current operation
            if isinstance(node.op, ast.Sub):
                from operator import sub
                res = map(sub, res_left, res_right)
            elif isinstance(node.op, ast.Add):
                from operator import add
                res = map(add, res_left, res_right)
            elif isinstance(node.op, ast.Mult):
                left_cnst = (res_left[:-1] == [0] * (dim - 1))
                right_cnst = (res_right[:-1] == [0] * (dim - 1))
                if not left_cnst and not right_cnst:
                    # Non-linear expression
                    raise Py2ScopException("ERROR: Unhandled operation MULT for non-constant expressions")
                elif not left_cnst:
                    # Right is constant
                    res = [x * res_right[-1] for x in res_left]
                elif not right_cnst:
                    # Left is constant
                    res = [x * res_left[-1] for x in res_right]
                else:
                    # Both constants
                    res = res_left
                    res[-1] = res[-1] * res_right[-1]
            elif isinstance(node.op, ast.Div):
                # Non-linear expression
                raise Py2ScopException("ERROR: Unhandled operation DIV for expressions")
        return res


#
# Exception Class
#

class Py2ScopException(Exception):

    def __init__(self, msg=None, nested_exception=None):
        self.msg = msg
        self.nested_exception = nested_exception

    def __str__(self):
        s = "Exception on Py2Scop.translate method.\n"
        if self.msg is not None:
            s = s + "Message: " + str(self.msg) + "\n"
        if self.nested_exception is not None:
            s = s + "Nested Exception: " + str(self.nested_exception) + "\n"
        return s


#
# UNIT TESTS
#

class TestPy2Scop(unittest.TestCase):

    @staticmethod
    def _test_ast_generation(func_name):
        # Insert function file into pythonpath
        import os
        dir_path = os.path.dirname(os.path.realpath(__file__))
        tests_path = dir_path + "/tests"
        import sys
        sys.path.insert(0, tests_path)

        # Import function to replace
        import importlib
        test_module = importlib.import_module("pycompss.util.translators.py2scop.tests.test1_ast_generation")
        func = getattr(test_module, func_name)

        # Create AST
        import inspect
        func_code = inspect.getsource(func)
        import ast
        func_ast = ast.parse(func_code)

        # Retrieve for blocks
        fbs = Py2Scop._ast_extract_for_blocks(func_ast)

        # DEBUG: Print fbs
        # print("---- DEBUG FOR " + str(func_name))
        # for fb in fbs:
        #        print(ast.dump(fb))

        # Return generated blocks
        return fbs

    @staticmethod
    def _test_ast2scop(func_name):
        # Insert function file into pythonpath
        import os
        dir_path = os.path.dirname(os.path.realpath(__file__))
        tests_path = dir_path + "/tests"
        import sys
        sys.path.insert(0, tests_path)

        # Import function to replace
        import importlib
        test_module = importlib.import_module("pycompss.util.translators.py2scop.tests.test2_ast2scop")
        func = getattr(test_module, func_name)

        # Create AST
        import inspect
        func_code = inspect.getsource(func)
        import ast
        func_ast = ast.parse(func_code)

        # Retrieve for blocks
        fbs = Py2Scop._ast_extract_for_blocks(func_ast, 0, [])

        # Apply SCOP transformation to first for block
        scop = Py2Scop._ast2scop(fbs[0], 0)

        # Return scop
        return scop

    def test_ast_empty(self):
        func_name = "empty"

        # Retrieve for blocks
        fbs = TestPy2Scop._test_ast_generation(func_name)

        # Check the number of generated for blocks
        self.assertEquals(fbs, None)

    def test_ast_simple1(self):
        func_name = "simple1"

        # Retrieve for blocks
        fbs = TestPy2Scop._test_ast_generation(func_name)

        # Check the number of generated for blocks
        self.assertEquals(len(fbs), 1)

    def test_ast_simple2(self):
        func_name = "simple2"

        # Retrieve for blocks
        fbs = TestPy2Scop._test_ast_generation(func_name)

        # Check the number of generated for blocks
        self.assertEquals(len(fbs), 1)

    def test_ast_simple3(self):
        func_name = "simple3"

        # Retrieve for blocks
        fbs = TestPy2Scop._test_ast_generation(func_name)

        # Check the number of generated for blocks
        self.assertEquals(len(fbs), 1)

    def test_ast_simple4(self):
        func_name = "simple4"

        # Retrieve for blocks
        fbs = TestPy2Scop._test_ast_generation(func_name)

        # Check the number of generated for blocks
        self.assertEquals(len(fbs), 1)

    def test_ast_intermediate1(self):
        func_name = "intermediate1"

        # Retrieve for blocks
        fbs = TestPy2Scop._test_ast_generation(func_name)

        # Check the number of generated for blocks
        self.assertEquals(len(fbs), 2)

    def test_ast_intermediate2(self):
        func_name = "intermediate2"

        # Retrieve for blocks
        fbs = TestPy2Scop._test_ast_generation(func_name)

        # Check the number of generated for blocks
        self.assertEquals(len(fbs), 2)

    def test_ast_intermediate3(self):
        func_name = "intermediate3"

        # Retrieve for blocks
        fbs = TestPy2Scop._test_ast_generation(func_name)

        # Check the number of generated for blocks
        self.assertEquals(len(fbs), 2)

    def test_ast_loop_nests1(self):
        func_name = "loop_nests1"

        # Retrieve for blocks
        fbs = TestPy2Scop._test_ast_generation(func_name)

        # Check the number of generated for blocks
        self.assertEquals(len(fbs), 1)

    def test_ast_loop_nests2(self):
        func_name = "loop_nests2"

        # Retrieve for blocks
        fbs = TestPy2Scop._test_ast_generation(func_name)

        # Check the number of generated for blocks
        self.assertEquals(len(fbs), 1)

    def test_ast_complex(self):
        func_name = "complex_loops"

        # Retrieve for blocks
        fbs = TestPy2Scop._test_ast_generation(func_name)

        # Check the number of generated for blocks
        self.assertEquals(len(fbs), 4)

    def test_ast2scop_simple1(self):
        func_name = "simple1"

        # Retrieve scop
        scop = TestPy2Scop._test_ast2scop(func_name)

        import os
        dir_path = os.path.dirname(os.path.realpath(__file__))
        test_file = dir_path + "/tests/" + str(func_name) + ".scop"
        try:
            # Write scop to file
            Py2Scop.write_os(scop, test_file)

            # Check scop
            expected_file = dir_path + "/tests/test2_ast2scop." + str(func_name) + ".expected.scop"
            with open(expected_file, 'r') as f:
                expected_content = f.read()
            with open(test_file, 'r') as f:
                out_content = f.read()
            self.assertEqual(out_content, expected_content)
        except Exception:
            raise
        finally:
            # Erase generated file
            os.remove(test_file)

    def test_ast2scop_simple2(self):
        func_name = "simple2"

        # Retrieve scop
        scop = TestPy2Scop._test_ast2scop(func_name)

        import os
        dir_path = os.path.dirname(os.path.realpath(__file__))
        test_file = dir_path + "/tests/" + str(func_name) + ".scop"
        try:
            # Write scop to file
            Py2Scop.write_os(scop, test_file)

            # Check scop
            expected_file = dir_path + "/tests/test2_ast2scop." + str(func_name) + ".expected.scop"
            with open(expected_file, 'r') as f:
                expected_content = f.read()
            with open(test_file, 'r') as f:
                out_content = f.read()
            self.assertEqual(out_content, expected_content)
        except Exception:
            raise
        finally:
            # Erase generated file
            os.remove(test_file)

    def test_ast2scop_simple3(self):
        func_name = "simple3"

        # Retrieve scop
        scop = TestPy2Scop._test_ast2scop(func_name)

        import os
        dir_path = os.path.dirname(os.path.realpath(__file__))
        test_file = dir_path + "/tests/" + str(func_name) + ".scop"
        try:
            # Write scop to file
            Py2Scop.write_os(scop, test_file)

            # Check scop
            expected_file = dir_path + "/tests/test2_ast2scop." + str(func_name) + ".expected.scop"
            with open(expected_file, 'r') as f:
                expected_content = f.read()
            with open(test_file, 'r') as f:
                out_content = f.read()
            self.assertEqual(out_content, expected_content)
        except Exception:
            raise
        finally:
            # Erase generated file
            os.remove(test_file)

    def test_ast2scop_simple4(self):
        func_name = "simple4"

        # Retrieve scop
        scop = TestPy2Scop._test_ast2scop(func_name)

        import os
        dir_path = os.path.dirname(os.path.realpath(__file__))
        test_file = dir_path + "/tests/" + str(func_name) + ".scop"
        try:
            # Write scop to file
            Py2Scop.write_os(scop, test_file)

            # Check scop
            expected_file = dir_path + "/tests/test2_ast2scop." + str(func_name) + ".expected.scop"
            with open(expected_file, 'r') as f:
                expected_content = f.read()
            with open(test_file, 'r') as f:
                out_content = f.read()
            self.assertEqual(out_content, expected_content)
        except Exception:
            raise
        finally:
            # Erase generated file
            os.remove(test_file)

    def test_ast2scop_simple5(self):
        func_name = "simple5"

        # Retrieve scop
        scop = TestPy2Scop._test_ast2scop(func_name)

        import os
        dir_path = os.path.dirname(os.path.realpath(__file__))
        test_file = dir_path + "/tests/" + str(func_name) + ".scop"
        try:
            # Write scop to file
            Py2Scop.write_os(scop, test_file)

            # Check scop
            expected_file = dir_path + "/tests/test2_ast2scop." + str(func_name) + ".expected.scop"
            with open(expected_file, 'r') as f:
                expected_content = f.read()
            with open(test_file, 'r') as f:
                out_content = f.read()
            self.assertEqual(out_content, expected_content)
        except Exception:
            raise
        finally:
            # Erase generated file
            os.remove(test_file)

    def test_ast2scop_loop_nests1(self):
        func_name = "loop_nests1"

        # Retrieve scop
        scop = TestPy2Scop._test_ast2scop(func_name)

        import os
        dir_path = os.path.dirname(os.path.realpath(__file__))
        test_file = dir_path + "/tests/" + str(func_name) + ".scop"
        try:
            # Write scop to file
            Py2Scop.write_os(scop, test_file)

            # Check scop
            expected_file = dir_path + "/tests/test2_ast2scop." + str(func_name) + ".expected.scop"
            with open(expected_file, 'r') as f:
                expected_content = f.read()
            with open(test_file, 'r') as f:
                out_content = f.read()
            self.assertEqual(out_content, expected_content)
        except Exception:
            raise
        finally:
            # Erase generated file
            os.remove(test_file)

    def test_ast2scop_loop_nests2(self):
        func_name = "loop_nests2"

        # Retrieve scop
        scop = TestPy2Scop._test_ast2scop(func_name)

        import os
        dir_path = os.path.dirname(os.path.realpath(__file__))
        test_file = dir_path + "/tests/" + str(func_name) + ".scop"
        try:
            # Write scop to file
            Py2Scop.write_os(scop, test_file)

            # Check scop
            expected_file = dir_path + "/tests/test2_ast2scop." + str(func_name) + ".expected.scop"
            with open(expected_file, 'r') as f:
                expected_content = f.read()
            with open(test_file, 'r') as f:
                out_content = f.read()
            self.assertEqual(out_content, expected_content)
        except Exception:
            raise
        finally:
            # Erase generated file
            os.remove(test_file)

    def test_ast2scop_multi_statements(self):
        func_name = "multi_statements"

        # Retrieve scop
        scop = TestPy2Scop._test_ast2scop(func_name)

        import os
        dir_path = os.path.dirname(os.path.realpath(__file__))
        test_file = dir_path + "/tests/" + str(func_name) + ".scop"
        try:
            # Write scop to file
            Py2Scop.write_os(scop, test_file)

            # Check scop
            expected_file = dir_path + "/tests/test2_ast2scop." + str(func_name) + ".expected.scop"
            with open(expected_file, 'r') as f:
                expected_content = f.read()
            with open(test_file, 'r') as f:
                out_content = f.read()
            self.assertEqual(out_content, expected_content)
        except Exception:
            raise
        finally:
            # Erase generated file
            os.remove(test_file)

    def test_ast2scop_func_call1(self):
        func_name = "func_call1"

        # Retrieve scop
        scop = TestPy2Scop._test_ast2scop(func_name)

        import os
        dir_path = os.path.dirname(os.path.realpath(__file__))
        test_file = dir_path + "/tests/" + str(func_name) + ".scop"
        try:
            # Write scop to file
            Py2Scop.write_os(scop, test_file)

            # Check scop
            expected_file = dir_path + "/tests/test2_ast2scop." + str(func_name) + ".expected.scop"
            with open(expected_file, 'r') as f:
                expected_content = f.read()
            with open(test_file, 'r') as f:
                out_content = f.read()
            self.assertEqual(out_content, expected_content)
        except Exception:
            raise
        finally:
            # Erase generated file
            os.remove(test_file)

    def test_ast2scop_func_call2(self):
        func_name = "func_call2"

        # Retrieve scop
        scop = TestPy2Scop._test_ast2scop(func_name)

        import os
        dir_path = os.path.dirname(os.path.realpath(__file__))
        test_file = dir_path + "/tests/" + str(func_name) + ".scop"
        try:
            # Write scop to file
            Py2Scop.write_os(scop, test_file)

            # Check scop
            expected_file = dir_path + "/tests/test2_ast2scop." + str(func_name) + ".expected.scop"
            with open(expected_file, 'r') as f:
                expected_content = f.read()
            with open(test_file, 'r') as f:
                out_content = f.read()
            self.assertEqual(out_content, expected_content)
        except Exception:
            raise
        finally:
            # Erase generated file
            os.remove(test_file)

    def test_ast2scop_func_call3(self):
        func_name = "func_call3"

        # Retrieve scop
        scop = TestPy2Scop._test_ast2scop(func_name)

        import os
        dir_path = os.path.dirname(os.path.realpath(__file__))
        test_file = dir_path + "/tests/" + str(func_name) + ".scop"
        try:
            # Write scop to file
            Py2Scop.write_os(scop, test_file)

            # Check scop
            expected_file = dir_path + "/tests/test2_ast2scop." + str(func_name) + ".expected.scop"
            with open(expected_file, 'r') as f:
                expected_content = f.read()
            with open(test_file, 'r') as f:
                out_content = f.read()
            self.assertEqual(out_content, expected_content)
        except Exception:
            raise
        finally:
            # Erase generated file
            os.remove(test_file)

    def test_ast2scop_func_call4(self):
        func_name = "func_call4"

        # Retrieve scop
        scop = TestPy2Scop._test_ast2scop(func_name)

        import os
        dir_path = os.path.dirname(os.path.realpath(__file__))
        test_file = dir_path + "/tests/" + str(func_name) + ".scop"
        try:
            # Write scop to file
            Py2Scop.write_os(scop, test_file)

            # Check scop
            expected_file = dir_path + "/tests/test2_ast2scop." + str(func_name) + ".expected.scop"
            with open(expected_file, 'r') as f:
                expected_content = f.read()
            with open(test_file, 'r') as f:
                out_content = f.read()
            self.assertEqual(out_content, expected_content)
        except Exception:
            raise
        finally:
            # Erase generated file
            os.remove(test_file)

    def test_ast2scop_func_call5(self):
        func_name = "func_call5"

        # Retrieve scop
        scop = TestPy2Scop._test_ast2scop(func_name)

        import os
        dir_path = os.path.dirname(os.path.realpath(__file__))
        test_file = dir_path + "/tests/" + str(func_name) + ".scop"
        try:
            # Write scop to file
            Py2Scop.write_os(scop, test_file)

            # Check scop
            expected_file = dir_path + "/tests/test2_ast2scop." + str(func_name) + ".expected.scop"
            with open(expected_file, 'r') as f:
                expected_content = f.read()
            with open(test_file, 'r') as f:
                out_content = f.read()
            self.assertEqual(out_content, expected_content)
        except Exception:
            raise
        finally:
            # Erase generated file
            os.remove(test_file)

    def test_matmul(self):
        import os
        dir_path = os.path.dirname(os.path.realpath(__file__))
        tests_path = dir_path + "/tests"

        func_name = "matmul"
        base_out_file = tests_path + "/test3_matmul.out.scop"

        try:
            # Insert function file into pythonpath
            import sys
            sys.path.insert(0, tests_path)

            # Import function to replace
            import importlib
            test_module = importlib.import_module("pycompss.util.translators.py2scop.tests.test3_matmul")
            func = getattr(test_module, func_name)

            # Translate
            translator = Py2Scop(func)
            output_files = translator.translate(base_out_file)

            # Check that there is only one output file
            self.assertEqual(len(output_files), 1)

            # Check file content
            expected_file = tests_path + "/test3_matmul.expected.scop"
            with open(expected_file, 'r') as f:
                expected_content = f.read()
            with open(output_files[0], 'r') as f:
                out_content = f.read()
            self.assertEqual(out_content, expected_content)
        except Exception:
            raise
        finally:
            # Erase generated files
            import glob
            for f in glob.glob(base_out_file + "*"):
                os.remove(f)


#
# MAIN
#

if __name__ == '__main__':
    logging.basicConfig(level=logging.DEBUG, format='%(asctime)s | %(levelname)s | %(name)s - %(message)s')
    unittest.main()
