#!/usr/bin/python

# -*- coding: utf-8 -*-

# For better print formatting
from __future__ import print_function

# Imports
from pycompss.api.parallel import parallel
from pycompss.api.constraint import constraint
from pycompss.api.task import task
from pycompss.api.api import compss_barrier
from pycompss.api.api import compss_wait_on

import numpy as np


############################################
# MATRIX GENERATION
############################################

def generate_matrix(m_size, b_size):
    mat = []
    for i in range(m_size):
        mat.append([])
        for _ in range(m_size):
            mb = create_block(b_size)
            mat[i].append(mb)
    return mat


@constraint(ComputingUnits="${ComputingUnits}")
@task(returns=list)
def create_block(b_size):
    import os
    np.random.seed(ord(os.urandom(1)))
    block = np.array(np.random.random((b_size, b_size)), dtype=np.float64, copy=False)
    mb = np.matrix(block, dtype=np.float64, copy=False)
    return mb


############################################
# MAIN FUNCTION
############################################

# [COMPSs Autoparallel] Begin Autogenerated code
import math

from pycompss.api.api import compss_barrier, compss_wait_on, compss_open
from pycompss.api.task import task
from pycompss.api.parameter import *


@task(b_size=IN, returns=1)
def S1(b_size):
    return np.matrix(np.zeros((b_size, b_size)), dtype=float)


@task(b_size=IN, returns=1)
def S2(b_size):
    return np.matrix(np.zeros((b_size, b_size)), dtype=float)


@task(var2=IN, returns=1)
def S3(var2):
    return invert_triangular(var2, lower=True)


@task(var2=IN, var3=IN, var4=IN, returns=1)
def S4(var2, var3, var4):
    return multiply([1], var2, var3, var4)


@task(var2=IN, returns=1)
def S5(var2):
    return invert_triangular(var2, lower=False)


@task(var2=IN, var3=IN, returns=1)
def S6(var2, var3):
    return multiply([], var2, var3)


@task(var2=IN, var3=IN, var4=IN, returns=1)
def S7(var2, var3, var4):
    return dgemm(-1, var2, var3, var4)


@task(var4=IN, returns=3)
def S8(var4):
    return custom_lu(var4)


@task(var2=IN, returns=1)
def S9(var2):
    return invert_triangular(var2, lower=False)


@task(var2=IN, var3=IN, var4=IN, returns=1)
def S10(var2, var3, var4):
    return multiply([0], var2, var3, var4)


@task(var2=IN, returns=1)
def S11(var2):
    return invert_triangular(var2, lower=True)


@task(var2=IN, var3=IN, var4=IN, returns=1)
def S12(var2, var3, var4):
    return multiply([1], var2, var3, var4)


def lu_blocked(a, m_size, b_size):
    if __debug__:
        a = compss_wait_on(a)
        print('Matrix A:')
        print(a)
    if __debug__:
        input_a = join_matrix(a)
        res_expected = np.zeros((m_size * b_size, m_size * b_size))
    if len(a) == 0:
        return
    p_mat = [([np.matrix(np.zeros((b_size, b_size)), dtype=float)] * m_size) for _ in range(m_size)]
    l_mat = [[None for _ in range(m_size)] for _ in range(m_size)]
    u_mat = [[None for _ in range(m_size)] for _ in range(m_size)]
    if m_size >= 2:
        lbp = 0
        ubp = m_size - 2
        for t1 in range(lbp, ubp + 1):
            lbv = t1 + 1
            ubv = m_size - 1
            for t2 in range(lbv, ubv + 1):
                l_mat[t1][t2] = S1(b_size)
                u_mat[t2][t1] = S2(b_size)
    compss_barrier()
    aux = [None]
    aux2 = [None]
    a[0][0] = compss_wait_on(a[0][0])
    p_mat[0][0], l_mat[0][0], u_mat[0][0] = custom_lu(a[0][0])
    if m_size >= 2:
        lbp = 1
        ubp = m_size - 1
        for t1 in range(1, m_size - 1 + 1):
            aux[0] = S3(l_mat[0][0])
            u_mat[0][t1] = S4(aux[0], p_mat[0][0], a[0][t1])
    compss_barrier()
    if m_size >= 2:
        lbp = 1
        ubp = m_size - 2
        for t1 in range(1, m_size - 2 + 1):
            lbp = t1
            ubp = m_size - 1
            for t3 in range(t1, m_size - 1 + 1):
                lbp = t1
                ubp = m_size - 1
                for t4 in range(t1, m_size - 1 + 1):
                    aux[0] = S5(u_mat[t1 - 1][t1 - 1])
                    aux2[0] = S6(a[t3][t1 - 1], aux[0])
                    a[t3][t4] = S7(a[t3][t4], aux2[0], u_mat[t1 - 1][t4])
            p_mat[t1][t1], l_mat[t1][t1], u_mat[t1][t1] = S8(a[t1][t1])
            lbp = 0
            ubp = t1 - 1
            for t3 in range(0, t1 - 1 + 1):
                aux[0] = S9(u_mat[t3][t3])
                l_mat[t1][t3] = S10(p_mat[t1][t1], a[t1][t3], aux[0])
            lbp = t1 + 1
            ubp = m_size - 1
            for t3 in range(t1 + 1, m_size - 1 + 1):
                aux[0] = S11(l_mat[t1][t1])
                u_mat[t1][t3] = S12(aux[0], p_mat[t1][t1], a[t1][t3])
        aux[0] = S5(u_mat[m_size - 1 - 1][m_size - 1 - 1])
        aux2[0] = S6(a[m_size - 1][m_size - 1 - 1], aux[0])
        a[m_size - 1][m_size - 1] = S7(a[m_size - 1][m_size - 1], aux2[0], u_mat[m_size - 1 - 1][m_size - 1])
        p_mat[m_size - 1][m_size - 1], l_mat[m_size - 1][m_size - 1], u_mat[m_size - 1][m_size - 1] = S8(a[m_size - 
            1][m_size - 1])
        lbp = 0
        ubp = m_size - 2
        for t3 in range(0, m_size - 2 + 1):
            aux[0] = S9(u_mat[t3][t3])
            l_mat[m_size - 1][t3] = S10(p_mat[m_size - 1][m_size - 1], a[m_size - 1][t3], aux[0])
    compss_barrier()
    if __debug__:
        p_res = join_matrix(compss_wait_on(p_mat))
        l_res = join_matrix(compss_wait_on(l_mat))
        u_res = join_matrix(compss_wait_on(u_mat))
        print('Matrix P:')
        print(p_res)
        print('Matrix L:')
        print(l_res)
        print('Matrix U:')
        print(u_res)
    if __debug__:
        check_result(input_a, p_res, l_res, u_res, res_expected)

# [COMPSs Autoparallel] End Autogenerated code


############################################
# MATHEMATICAL FUNCTIONS
############################################

def invert_triangular(mat, lower=False):
    from scipy.linalg import solve_triangular

    print(mat)

    dim = len(mat)
    iden = np.matrix(np.identity(dim))
    return solve_triangular(mat, iden, lower=lower)


def multiply(inv_list, *args):
    assert len(args) > 0

    input_args = list(args)
    if len(inv_list) > 0:
        from numpy.linalg import inv
        for elem in inv_list:
            input_args[elem] = inv(args[elem])

    if len(input_args) == 1:
        return input_args[0]

    result = np.dot(input_args[0], input_args[1])
    for i in range(2, len(input_args)):
        result = np.dot(result, input_args[i])
    return result


def dgemm(alpha, a, b, c):
    mat = a + (alpha * np.dot(b, c))
    return mat


def custom_lu(mat):
    from scipy.linalg import lu

    return lu(mat)


############################################
# BLOCK HANDLING FUNCTIONS
############################################

def join_matrix(mat):
    joint_mat = np.matrix([[]])
    for i in range(0, len(mat)):
        current_row = mat[i][0]
        for j in range(1, len(mat[i])):
            current_row = np.bmat([[current_row, mat[i][j]]])
        if i == 0:
            joint_mat = current_row
        else:
            joint_mat = np.bmat([[joint_mat], [current_row]])

    return np.matrix(joint_mat)


def check_result(input_a, p_res, l_res, u_res, result_expected):
    result = input_a - np.dot(np.dot(p_res, l_res), u_res)
    is_ok = np.allclose(result, result_expected)
    print("Result check status: " + str(is_ok))

    if not is_ok:
        raise Exception("Result does not match expected result")


############################################
# MAIN
############################################

if __name__ == "__main__":
    # Import libraries
    import time

    # Parse arguments
    import sys

    args = sys.argv[1:]
    MSIZE = int(args[0])
    BSIZE = int(args[1])

    # Log arguments if required
    if __debug__:
        print("Running LU application with:")
        print(" - MSIZE = " + str(MSIZE))
        print(" - BSIZE = " + str(BSIZE))

    # Initialize matrix
    if __debug__:
        print("Initializing matrix")
    start_time = time.time()
    A = generate_matrix(MSIZE, BSIZE)
    compss_barrier()

    # Begin computation
    if __debug__:
        print("Performing computation")
    lu_start_time = time.time()
    lu_blocked(A, MSIZE, BSIZE)
    compss_barrier(True)
    end_time = time.time()

    # Log results and time
    if __debug__:
        print("Post-process results")
    total_time = end_time - start_time
    init_time = lu_start_time - start_time
    lu_time = end_time - lu_start_time

    print("RESULTS -----------------")
    print("VERSION AUTOPARALLEL")
    print("MSIZE " + str(MSIZE))
    print("BSIZE " + str(BSIZE))
    print("DEBUG " + str(__debug__))
    print("TOTAL_TIME " + str(total_time))
    print("INIT_TIME " + str(init_time))
    print("LU_TIME " + str(lu_time))
    print("-------------------------")
