#!/usr/bin/python

# -*- coding: utf-8 -*-

# For better print formatting
from __future__ import print_function

# Imports
from pycompss.api.parallel import parallel
from pycompss.api.constraint import constraint
from pycompss.api.task import task
from pycompss.api.api import compss_barrier
from pycompss.api.api import compss_wait_on

import numpy as np


############################################
# MATRIX GENERATION
############################################

def generate_matrix(m_size, b_size):
    mat = []
    for i in range(m_size):
        mat.append([])
        for _ in range(m_size):
            mat[i].append([])

    for i in range(m_size):
        mat[i][i] = create_block(b_size, True)
        for j in range(i + 1, m_size):
            mat[i][j] = create_block(b_size, False)

    # Make it symmetric
    for i in range(m_size):
        mat[i][i] = compss_wait_on(mat[i][i])
        for j in range(i + 1, m_size):
            mat[i][j] = compss_wait_on(mat[i][j])  # To break aliasing between future objects
            mat[j][i] = mat[i][j]

    return mat


@constraint(ComputingUnits="${ComputingUnits}")
@task(returns=list)
def create_block(b_size, is_diag):
    block = np.array(np.random.random((b_size, b_size)), dtype=np.float64, copy=False)
    mb = np.matrix(block, dtype=np.float64, copy=False)
    mb = mb + np.transpose(mb)
    if is_diag:
        mb = mb + 2 * b_size * np.eye(b_size)
    return mb


############################################
# MAIN FUNCTION
############################################

# [COMPSs Autoparallel] Begin Autogenerated code
import math

from pycompss.api.api import compss_barrier, compss_wait_on, compss_open
from pycompss.api.task import task
from pycompss.api.parameter import *


@task(var2=IN, returns=1)
def S1(var2):
    return potrf(var2)


@task(var2=IN, var3=IN, returns=1)
def S2(var2, var3):
    return solve_triangular(var2, var3)


@task(b_size=IN, returns=1)
def S3(b_size):
    return np.zeros((b_size, b_size))


@task(var2=IN, var3=IN, var4=IN, returns=1)
def S4(var2, var3, var4):
    return gemm(-1.0, var2, var3, var4, 1.0)


def cholesky_blocked(a, m_size, b_size):
    if __debug__:
        a = compss_wait_on(a)
        print('Matrix A:')
        print(a)
    if __debug__:
        from numpy.linalg import cholesky as cholesky_numpy
        res_expected = cholesky_numpy(join_matrix(a))
    if m_size >= 1:
        a[0][0] = S1(a[0][0])
        lbp = 1
        ubp = min(2, m_size - 1)
        for t1 in range(1, min(2, m_size - 1) + 1):
            if t1 == 2:
                a[1][1] = S1(a[1][1])
            a[t1][0] = S2(a[0][0], a[t1][0])
            a[0][t1] = S3(b_size)
            lbp = 1
            ubp = t1
            for t3 in range(1, t1 + 1):
                a[t1][t3] = S4(a[t1][0], a[t3][0], a[t1][t3])
        lbp = 3
        ubp = m_size - 1
        for t1 in range(3, m_size - 1 + 1):
            if t1 % 2 == 0:
                a[t1 / 2][t1 / 2] = S1(a[t1 / 2][t1 / 2])
            lbp = int(math.ceil(float(t1 + 1) / float(2)))
            ubp = t1 - 1
            for t2 in range(lbp, ubp + 1):
                a[t1 - t2][t2] = S3(b_size)
                a[t2][t1 - t2] = S2(a[t1 - t2][t1 - t2], a[t2][t1 - t2])
                lbp = t1 - t2 + 1
                ubp = t2
                for t3 in range(t1 - t2 + 1, t2 + 1):
                    a[t2][t3] = S4(a[t2][t1 - t2], a[t3][t1 - t2], a[t2][t3])
            a[t1][0] = S2(a[0][0], a[t1][0])
            a[0][t1] = S3(b_size)
            lbp = 1
            ubp = t1
            for t3 in range(1, t1 + 1):
                a[t1][t3] = S4(a[t1][0], a[t3][0], a[t1][t3])
        lbp = m_size
        ubp = 2 * m_size - 3
        for t1 in range(m_size, 2 * m_size - 3 + 1):
            if t1 % 2 == 0:
                a[t1 / 2][t1 / 2] = S1(a[t1 / 2][t1 / 2])
            lbp = int(math.ceil(float(t1 + 1) / float(2)))
            ubp = m_size - 1
            for t2 in range(lbp, ubp + 1):
                a[t1 - t2][t2] = S3(b_size)
                a[t2][t1 - t2] = S2(a[t1 - t2][t1 - t2], a[t2][t1 - t2])
                lbp = t1 - t2 + 1
                ubp = t2
                for t3 in range(t1 - t2 + 1, t2 + 1):
                    a[t2][t3] = S4(a[t2][t1 - t2], a[t3][t1 - t2], a[t2][t3])
        if m_size >= 2:
            a[m_size - 1][m_size - 1] = S1(a[m_size - 1][m_size - 1])
    compss_barrier()
    if __debug__:
        a = compss_wait_on(a)
        res = join_matrix(a)
        print('New Matrix A:')
        print(res)
    if __debug__:
        check_result(res, res_expected)

# [COMPSs Autoparallel] End Autogenerated code


############################################
# MATHEMATICAL FUNCTIONS
############################################

def potrf(a):
    from scipy.linalg.lapack import dpotrf
    a = dpotrf(a, lower=True)[0]
    return a


def solve_triangular(a, b):
    from scipy.linalg import solve_triangular
    from numpy import transpose

    b = transpose(b)
    b = solve_triangular(a, b, lower=True)
    b = transpose(b)
    return b


def gemm(alpha, a, b, c, beta):
    from scipy.linalg.blas import dgemm
    from numpy import transpose

    b = transpose(b)
    c = dgemm(alpha, a, b, c=c, beta=beta)
    return c


def syrk(a, b):
    from scipy.linalg.blas import dsyrk

    alpha = -1.0
    beta = 1.0
    b = dsyrk(alpha, a, c=b, beta=beta, lower=True)
    return b


############################################
# BLOCK HANDLING FUNCTIONS
############################################

def join_matrix(a):
    joint_matrix = np.matrix([[]])
    for i in range(0, len(a)):
        current_row = a[i][0]
        for j in range(1, len(a[i])):
            current_row = np.bmat([[current_row, a[i][j]]])
        if i == 0:
            joint_matrix = current_row
        else:
            joint_matrix = np.bmat([[joint_matrix], [current_row]])

    return np.matrix(joint_matrix)


def check_result(result, result_expected):
    is_ok = np.allclose(result, result_expected)
    print("Result check status: " + str(is_ok))

    if not is_ok:
        raise Exception("Result does not match expected result")


############################################
# MAIN
############################################

if __name__ == "__main__":
    # Import libraries
    import time

    # Parse arguments
    import sys

    args = sys.argv[1:]
    MSIZE = int(args[0])
    BSIZE = int(args[1])

    # Log arguments if required
    if __debug__:
        print("Running cholesky application with:")
        print(" - MSIZE = " + str(MSIZE))
        print(" - BSIZE = " + str(BSIZE))

    # Initialize matrix
    if __debug__:
        print("Initializing matrix")
    start_time = time.time()
    A = generate_matrix(MSIZE, BSIZE)
    compss_barrier()

    # Begin computation
    if __debug__:
        print("Performing computation")
    cholesky_start_time = time.time()
    cholesky_blocked(A, MSIZE, BSIZE)
    compss_barrier(True)
    end_time = time.time()

    # Log results and time
    if __debug__:
        print("Post-process results")
    total_time = end_time - start_time
    init_time = cholesky_start_time - start_time
    cholesky_time = end_time - cholesky_start_time

    print("RESULTS -----------------")
    print("VERSION AUTOPARALLEL")
    print("MSIZE " + str(MSIZE))
    print("BSIZE " + str(BSIZE))
    print("DEBUG " + str(__debug__))
    print("TOTAL_TIME " + str(total_time))
    print("INIT_TIME " + str(init_time))
    print("CHOLESKY_TIME " + str(cholesky_time))
    print("-------------------------")
