/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.message;

import java.io.Writer;
import java.util.HashSet;
import java.util.Optional;
import java.util.stream.Collectors;
import org.apache.kafka.message.CodeBuffer;
import org.apache.kafka.message.FieldSpec;
import org.apache.kafka.message.FieldType;
import org.apache.kafka.message.HeaderGenerator;
import org.apache.kafka.message.MessageGenerator;
import org.apache.kafka.message.MessageSpec;
import org.apache.kafka.message.SchemaGenerator;
import org.apache.kafka.message.StructSpec;
import org.apache.kafka.message.Versions;

public final class MessageDataGenerator {
    private final HeaderGenerator headerGenerator = new HeaderGenerator();
    private final SchemaGenerator schemaGenerator = new SchemaGenerator(this.headerGenerator);
    private final CodeBuffer buffer = new CodeBuffer();

    MessageDataGenerator() {
    }

    void generate(MessageSpec message) throws Exception {
        if (message.struct().versions().contains((short)Short.MAX_VALUE)) {
            throw new RuntimeException("Message " + message.name() + " does not specify a maximum version.");
        }
        this.schemaGenerator.generateSchemas(message);
        this.generateClass(Optional.of(message), message.name() + "Data", message.struct(), message.struct().versions());
        this.headerGenerator.generate();
    }

    void write(Writer writer) throws Exception {
        this.headerGenerator.buffer().write(writer);
        this.buffer.write(writer);
    }

    private void generateClass(Optional<MessageSpec> topLevelMessageSpec, String className, StructSpec struct, Versions parentVersions) throws Exception {
        this.buffer.printf("%n", new Object[0]);
        boolean isTopLevel = topLevelMessageSpec.isPresent();
        boolean isSetElement = struct.hasKeys();
        if (isTopLevel && isSetElement) {
            throw new RuntimeException("Cannot set mapKey on top level fields.");
        }
        this.generateClassHeader(className, isTopLevel, isSetElement);
        this.buffer.incrementIndent();
        this.generateFieldDeclarations(struct, isSetElement);
        this.buffer.printf("%n", new Object[0]);
        this.schemaGenerator.writeSchema(className, this.buffer);
        this.generateClassConstructors(className, struct);
        this.buffer.printf("%n", new Object[0]);
        if (isTopLevel) {
            this.generateShortAccessor("apiKey", topLevelMessageSpec.get().apiKey().orElse((short)-1));
        }
        this.buffer.printf("%n", new Object[0]);
        this.generateShortAccessor("lowestSupportedVersion", parentVersions.lowest());
        this.buffer.printf("%n", new Object[0]);
        this.generateShortAccessor("highestSupportedVersion", parentVersions.highest());
        this.buffer.printf("%n", new Object[0]);
        this.generateClassReader(className, struct, parentVersions);
        this.buffer.printf("%n", new Object[0]);
        this.generateClassWriter(className, struct, parentVersions);
        this.buffer.printf("%n", new Object[0]);
        this.generateClassFromStruct(className, struct, parentVersions);
        this.buffer.printf("%n", new Object[0]);
        this.generateClassToStruct(className, struct, parentVersions);
        this.buffer.printf("%n", new Object[0]);
        this.generateClassSize(className, struct, parentVersions);
        this.buffer.printf("%n", new Object[0]);
        this.generateClassEquals(className, struct, isSetElement);
        this.buffer.printf("%n", new Object[0]);
        this.generateClassHashCode(struct, isSetElement);
        this.buffer.printf("%n", new Object[0]);
        this.generateClassToString(className, struct);
        this.generateFieldAccessors(struct, isSetElement);
        this.generateFieldMutators(struct, className, isSetElement);
        if (!isTopLevel) {
            this.buffer.decrementIndent();
            this.buffer.printf("}%n", new Object[0]);
        }
        this.generateSubclasses(className, struct, parentVersions, isSetElement);
        if (isTopLevel) {
            this.buffer.decrementIndent();
            this.buffer.printf("}%n", new Object[0]);
        }
    }

    private void generateClassHeader(String className, boolean isTopLevel, boolean isSetElement) {
        HashSet<String> implementedInterfaces = new HashSet<String>();
        if (isTopLevel) {
            implementedInterfaces.add("ApiMessage");
            this.headerGenerator.addImport("org.apache.kafka.common.protocol.ApiMessage");
        } else {
            implementedInterfaces.add("Message");
            this.headerGenerator.addImport("org.apache.kafka.common.protocol.Message");
        }
        if (isSetElement) {
            this.headerGenerator.addImport("org.apache.kafka.common.utils.ImplicitLinkedHashMultiCollection");
            implementedInterfaces.add("ImplicitLinkedHashMultiCollection.Element");
        }
        HashSet<String> classModifiers = new HashSet<String>();
        classModifiers.add("public");
        if (!isTopLevel) {
            classModifiers.add("static");
        }
        this.buffer.printf("%s class %s implements %s {%n", String.join((CharSequence)" ", classModifiers), className, String.join((CharSequence)", ", implementedInterfaces));
    }

    private void generateSubclasses(String className, StructSpec struct, Versions parentVersions, boolean isSetElement) throws Exception {
        for (FieldSpec field : struct.fields()) {
            if (!field.type().isStructArray()) continue;
            FieldType.ArrayType arrayType = (FieldType.ArrayType)field.type();
            this.generateClass(Optional.empty(), arrayType.elementType().toString(), field.toStruct(), parentVersions.intersect(struct.versions()));
        }
        if (isSetElement) {
            this.generateHashSet(className, struct);
        }
    }

    private void generateHashSet(String className, StructSpec struct) {
        this.buffer.printf("%n", new Object[0]);
        this.headerGenerator.addImport("org.apache.kafka.common.utils.ImplicitLinkedHashMultiCollection");
        this.buffer.printf("public static class %s extends ImplicitLinkedHashMultiCollection<%s> {%n", MessageDataGenerator.collectionType(className), className);
        this.buffer.incrementIndent();
        this.generateHashSetZeroArgConstructor(className);
        this.generateHashSetSizeArgConstructor(className);
        this.generateHashSetIteratorConstructor(className);
        this.generateHashSetFindMethod(className, struct);
        this.generateHashSetFindAllMethod(className, struct);
        this.buffer.decrementIndent();
        this.buffer.printf("}%n", new Object[0]);
    }

    private void generateHashSetZeroArgConstructor(String className) {
        this.buffer.printf("public %s() {%n", MessageDataGenerator.collectionType(className));
        this.buffer.incrementIndent();
        this.buffer.printf("super();%n", new Object[0]);
        this.buffer.decrementIndent();
        this.buffer.printf("}%n", new Object[0]);
        this.buffer.printf("%n", new Object[0]);
    }

    private void generateHashSetSizeArgConstructor(String className) {
        this.buffer.printf("public %s(int expectedNumElements) {%n", MessageDataGenerator.collectionType(className));
        this.buffer.incrementIndent();
        this.buffer.printf("super(expectedNumElements);%n", new Object[0]);
        this.buffer.decrementIndent();
        this.buffer.printf("}%n", new Object[0]);
        this.buffer.printf("%n", new Object[0]);
    }

    private void generateHashSetIteratorConstructor(String className) {
        this.headerGenerator.addImport("java.util.Iterator");
        this.buffer.printf("public %s(Iterator<%s> iterator) {%n", MessageDataGenerator.collectionType(className), className);
        this.buffer.incrementIndent();
        this.buffer.printf("super(iterator);%n", new Object[0]);
        this.buffer.decrementIndent();
        this.buffer.printf("}%n", new Object[0]);
        this.buffer.printf("%n", new Object[0]);
    }

    private void generateHashSetFindMethod(String className, StructSpec struct) {
        this.headerGenerator.addImport("java.util.List");
        this.buffer.printf("public %s find(%s) {%n", className, this.commaSeparatedHashSetFieldAndTypes(struct));
        this.buffer.incrementIndent();
        this.generateKeyElement(className, struct);
        this.headerGenerator.addImport("org.apache.kafka.common.utils.ImplicitLinkedHashMultiCollection");
        this.buffer.printf("return find(key);%n", new Object[0]);
        this.buffer.decrementIndent();
        this.buffer.printf("}%n", new Object[0]);
        this.buffer.printf("%n", new Object[0]);
    }

    private void generateHashSetFindAllMethod(String className, StructSpec struct) {
        this.headerGenerator.addImport("java.util.List");
        this.buffer.printf("public List<%s> findAll(%s) {%n", className, this.commaSeparatedHashSetFieldAndTypes(struct));
        this.buffer.incrementIndent();
        this.generateKeyElement(className, struct);
        this.headerGenerator.addImport("org.apache.kafka.common.utils.ImplicitLinkedHashMultiCollection");
        this.buffer.printf("return findAll(key);%n", new Object[0]);
        this.buffer.decrementIndent();
        this.buffer.printf("}%n", new Object[0]);
        this.buffer.printf("%n", new Object[0]);
    }

    private void generateKeyElement(String className, StructSpec struct) {
        this.buffer.printf("%s key = new %s();%n", className, className);
        for (FieldSpec field : struct.fields()) {
            if (!field.mapKey()) continue;
            this.buffer.printf("key.set%s(%s);%n", field.capitalizedCamelCaseName(), field.camelCaseName());
        }
    }

    private String commaSeparatedHashSetFieldAndTypes(StructSpec struct) {
        return struct.fields().stream().filter(f -> f.mapKey()).map(f -> String.format("%s %s", this.fieldConcreteJavaType((FieldSpec)f), f.camelCaseName())).collect(Collectors.joining(", "));
    }

    private void generateFieldDeclarations(StructSpec struct, boolean isSetElement) {
        for (FieldSpec field : struct.fields()) {
            this.generateFieldDeclaration(field);
        }
        if (isSetElement) {
            this.buffer.printf("private int next;%n", new Object[0]);
            this.buffer.printf("private int prev;%n", new Object[0]);
        }
    }

    private void generateFieldDeclaration(FieldSpec field) {
        this.buffer.printf("private %s %s;%n", this.fieldAbstractJavaType(field), field.camelCaseName());
    }

    private void generateFieldAccessors(StructSpec struct, boolean isSetElement) {
        for (FieldSpec field : struct.fields()) {
            this.generateFieldAccessor(field);
        }
        if (isSetElement) {
            this.buffer.printf("%n", new Object[0]);
            this.buffer.printf("@Override%n", new Object[0]);
            this.generateAccessor("int", "next", "next");
            this.buffer.printf("%n", new Object[0]);
            this.buffer.printf("@Override%n", new Object[0]);
            this.generateAccessor("int", "prev", "prev");
        }
    }

    private void generateFieldMutators(StructSpec struct, String className, boolean isSetElement) {
        for (FieldSpec field : struct.fields()) {
            this.generateFieldMutator(className, field);
        }
        if (isSetElement) {
            this.buffer.printf("%n", new Object[0]);
            this.buffer.printf("@Override%n", new Object[0]);
            this.generateSetter("int", "setNext", "next");
            this.buffer.printf("%n", new Object[0]);
            this.buffer.printf("@Override%n", new Object[0]);
            this.generateSetter("int", "setPrev", "prev");
        }
    }

    private static String collectionType(String baseType) {
        return baseType + "Collection";
    }

    private String fieldAbstractJavaType(FieldSpec field) {
        if (field.type() instanceof FieldType.BoolFieldType) {
            return "boolean";
        }
        if (field.type() instanceof FieldType.Int8FieldType) {
            return "byte";
        }
        if (field.type() instanceof FieldType.Int16FieldType) {
            return "short";
        }
        if (field.type() instanceof FieldType.Int32FieldType) {
            return "int";
        }
        if (field.type() instanceof FieldType.Int64FieldType) {
            return "long";
        }
        if (field.type().isString()) {
            return "String";
        }
        if (field.type().isBytes()) {
            return "byte[]";
        }
        if (field.type().isStruct()) {
            return MessageGenerator.capitalizeFirst(field.typeString());
        }
        if (field.type().isArray()) {
            FieldType.ArrayType arrayType = (FieldType.ArrayType)field.type();
            if (field.toStruct().hasKeys()) {
                this.headerGenerator.addImport("org.apache.kafka.common.utils.ImplicitLinkedHashMultiCollection");
                return MessageDataGenerator.collectionType(arrayType.elementType().toString());
            }
            this.headerGenerator.addImport("java.util.List");
            return "List<" + this.getBoxedJavaType(arrayType.elementType()) + ">";
        }
        throw new RuntimeException("Unknown field type " + field.type());
    }

    private String fieldConcreteJavaType(FieldSpec field) {
        if (field.type().isArray()) {
            FieldType.ArrayType arrayType = (FieldType.ArrayType)field.type();
            if (field.toStruct().hasKeys()) {
                this.headerGenerator.addImport("org.apache.kafka.common.utils.ImplicitLinkedHashMultiCollection");
                return MessageDataGenerator.collectionType(arrayType.elementType().toString());
            }
            this.headerGenerator.addImport("java.util.ArrayList");
            return "ArrayList<" + this.getBoxedJavaType(arrayType.elementType()) + ">";
        }
        return this.fieldAbstractJavaType(field);
    }

    private void generateClassConstructors(String className, StructSpec struct) {
        this.headerGenerator.addImport("org.apache.kafka.common.protocol.Readable");
        this.buffer.printf("public %s(Readable readable, short version) {%n", className);
        this.buffer.incrementIndent();
        this.initializeArrayDefaults(struct);
        this.buffer.printf("read(readable, version);%n", new Object[0]);
        this.buffer.decrementIndent();
        this.buffer.printf("}%n", new Object[0]);
        this.buffer.printf("%n", new Object[0]);
        this.headerGenerator.addImport("org.apache.kafka.common.protocol.types.Struct");
        this.buffer.printf("public %s(Struct struct, short version) {%n", className);
        this.buffer.incrementIndent();
        this.initializeArrayDefaults(struct);
        this.buffer.printf("fromStruct(struct, version);%n", new Object[0]);
        this.buffer.decrementIndent();
        this.buffer.printf("}%n", new Object[0]);
        this.buffer.printf("%n", new Object[0]);
        this.buffer.printf("public %s() {%n", className);
        this.buffer.incrementIndent();
        for (FieldSpec field : struct.fields()) {
            this.buffer.printf("this.%s = %s;%n", field.camelCaseName(), this.fieldDefault(field));
        }
        this.buffer.decrementIndent();
        this.buffer.printf("}%n", new Object[0]);
    }

    private void initializeArrayDefaults(StructSpec struct) {
        for (FieldSpec field : struct.fields()) {
            if (!field.type().isArray()) continue;
            this.buffer.printf("this.%s = %s;%n", field.camelCaseName(), this.fieldDefault(field));
        }
    }

    private void generateShortAccessor(String name, short val) {
        this.buffer.printf("@Override%n", new Object[0]);
        this.buffer.printf("public short %s() {%n", name);
        this.buffer.incrementIndent();
        this.buffer.printf("return %d;%n", val);
        this.buffer.decrementIndent();
        this.buffer.printf("}%n", new Object[0]);
    }

    private void generateClassReader(String className, StructSpec struct, Versions parentVersions) {
        Versions curVersions;
        this.headerGenerator.addImport("org.apache.kafka.common.protocol.Readable");
        this.buffer.printf("@Override%n", new Object[0]);
        this.buffer.printf("public void read(Readable readable, short version) {%n", new Object[0]);
        this.buffer.incrementIndent();
        if (this.generateInverseVersionCheck(parentVersions, struct.versions())) {
            this.buffer.incrementIndent();
            this.headerGenerator.addImport("org.apache.kafka.common.errors.UnsupportedVersionException");
            this.buffer.printf("throw new UnsupportedVersionException(\"Can't read version \" + version + \" of %s\");%n", className);
            this.buffer.decrementIndent();
            this.buffer.printf("}%n", new Object[0]);
        }
        if ((curVersions = parentVersions.intersect(struct.versions())).empty()) {
            throw new RuntimeException("Version ranges " + parentVersions + " and " + struct.versions() + " have no versions in common.");
        }
        for (FieldSpec field : struct.fields()) {
            this.generateFieldReader(field, curVersions);
        }
        this.buffer.decrementIndent();
        this.buffer.printf("}%n", new Object[0]);
    }

    private void generateFieldReader(FieldSpec field, Versions curVersions) {
        if (field.type().isArray()) {
            boolean maybeAbsent = this.generateVersionCheck(curVersions, field.versions());
            if (!maybeAbsent) {
                this.buffer.printf("{%n", new Object[0]);
                this.buffer.incrementIndent();
            }
            boolean hasKeys = field.toStruct().hasKeys();
            this.buffer.printf("int arrayLength = readable.readInt();%n", new Object[0]);
            this.buffer.printf("if (arrayLength < 0) {%n", new Object[0]);
            this.buffer.incrementIndent();
            this.buffer.printf("this.%s = null;%n", field.camelCaseName());
            this.buffer.decrementIndent();
            this.buffer.printf("} else {%n", new Object[0]);
            this.buffer.incrementIndent();
            this.buffer.printf("this.%s.clear(%s);%n", field.camelCaseName(), hasKeys ? "arrayLength" : "");
            this.buffer.printf("for (int i = 0; i < arrayLength; i++) {%n", new Object[0]);
            this.buffer.incrementIndent();
            FieldType.ArrayType arrayType = (FieldType.ArrayType)field.type();
            this.buffer.printf("this.%s.add(%s);%n", field.camelCaseName(), this.readFieldFromReadable(arrayType.elementType()));
            this.buffer.decrementIndent();
            this.buffer.printf("}%n", new Object[0]);
            this.buffer.decrementIndent();
            this.buffer.printf("}%n", new Object[0]);
            if (maybeAbsent) {
                this.generateSetDefault(field);
            } else {
                this.buffer.decrementIndent();
                this.buffer.printf("}%n", new Object[0]);
            }
        } else {
            boolean maybeAbsent = this.generateVersionCheck(curVersions, field.versions());
            this.buffer.printf("this.%s = %s;%n", field.camelCaseName(), this.readFieldFromReadable(field.type()));
            if (maybeAbsent) {
                this.generateSetDefault(field);
            }
        }
    }

    private String readFieldFromReadable(FieldType type) {
        if (type instanceof FieldType.BoolFieldType) {
            return "readable.readByte() != 0";
        }
        if (type instanceof FieldType.Int8FieldType) {
            return "readable.readByte()";
        }
        if (type instanceof FieldType.Int16FieldType) {
            return "readable.readShort()";
        }
        if (type instanceof FieldType.Int32FieldType) {
            return "readable.readInt()";
        }
        if (type instanceof FieldType.Int64FieldType) {
            return "readable.readLong()";
        }
        if (type.isString()) {
            return "readable.readNullableString()";
        }
        if (type.isBytes()) {
            return "readable.readNullableBytes()";
        }
        if (type.isStruct()) {
            return String.format("new %s(readable, version)", type.toString());
        }
        throw new RuntimeException("Unsupported field type " + type);
    }

    private void generateClassFromStruct(String className, StructSpec struct, Versions parentVersions) {
        Versions curVersions;
        this.headerGenerator.addImport("org.apache.kafka.common.protocol.types.Struct");
        this.buffer.printf("@Override%n", new Object[0]);
        this.buffer.printf("public void fromStruct(Struct struct, short version) {%n", new Object[0]);
        this.buffer.incrementIndent();
        if (this.generateInverseVersionCheck(parentVersions, struct.versions())) {
            this.buffer.incrementIndent();
            this.headerGenerator.addImport("org.apache.kafka.common.errors.UnsupportedVersionException");
            this.buffer.printf("throw new UnsupportedVersionException(\"Can't read version \" + version + \" of %s\");%n", className);
            this.buffer.decrementIndent();
            this.buffer.printf("}%n", new Object[0]);
        }
        if ((curVersions = parentVersions.intersect(struct.versions())).empty()) {
            throw new RuntimeException("Version ranges " + parentVersions + " and " + struct.versions() + " have no versions in common.");
        }
        for (FieldSpec field : struct.fields()) {
            this.generateFieldFromStruct(field, curVersions);
        }
        this.buffer.decrementIndent();
        this.buffer.printf("}%n", new Object[0]);
    }

    private void generateFieldFromStruct(FieldSpec field, Versions curVersions) {
        if (field.type().isArray()) {
            boolean maybeAbsent = this.generateVersionCheck(curVersions, field.versions());
            if (!maybeAbsent) {
                this.buffer.printf("{%n", new Object[0]);
                this.buffer.incrementIndent();
            }
            this.headerGenerator.addImport("org.apache.kafka.common.protocol.types.Struct");
            this.buffer.printf("Object[] nestedObjects = struct.getArray(\"%s\");%n", field.snakeCaseName());
            boolean maybeNull = false;
            if (!curVersions.intersect(field.nullableVersions()).empty()) {
                maybeNull = true;
                this.buffer.printf("if (nestedObjects == null) {%n", field.camelCaseName());
                this.buffer.incrementIndent();
                this.buffer.printf("this.%s = null;%n", field.camelCaseName());
                this.buffer.decrementIndent();
                this.buffer.printf("} else {%n", new Object[0]);
                this.buffer.incrementIndent();
            }
            FieldType.ArrayType arrayType = (FieldType.ArrayType)field.type();
            FieldType elementType = arrayType.elementType();
            this.buffer.printf("this.%s = new %s(nestedObjects.length);%n", field.camelCaseName(), this.fieldConcreteJavaType(field));
            this.buffer.printf("for (Object nestedObject : nestedObjects) {%n", new Object[0]);
            this.buffer.incrementIndent();
            if (elementType.isStruct()) {
                this.buffer.printf("this.%s.add(new %s((Struct) nestedObject, version));%n", field.camelCaseName(), elementType.toString());
            } else {
                this.buffer.printf("this.%s.add((%s) nestedObject);%n", field.camelCaseName(), this.getBoxedJavaType(elementType));
            }
            this.buffer.decrementIndent();
            this.buffer.printf("}%n", new Object[0]);
            if (maybeNull) {
                this.buffer.decrementIndent();
                this.buffer.printf("}%n", new Object[0]);
            }
            if (maybeAbsent) {
                this.generateSetDefault(field);
            } else {
                this.buffer.decrementIndent();
                this.buffer.printf("}%n", new Object[0]);
            }
        } else {
            boolean maybeAbsent = this.generateVersionCheck(curVersions, field.versions());
            this.buffer.printf("this.%s = %s;%n", field.camelCaseName(), this.readFieldFromStruct(field.type(), field.snakeCaseName()));
            if (maybeAbsent) {
                this.generateSetDefault(field);
            }
        }
    }

    private String getBoxedJavaType(FieldType type) {
        if (type instanceof FieldType.BoolFieldType) {
            return "Boolean";
        }
        if (type instanceof FieldType.Int8FieldType) {
            return "Byte";
        }
        if (type instanceof FieldType.Int16FieldType) {
            return "Short";
        }
        if (type instanceof FieldType.Int32FieldType) {
            return "Integer";
        }
        if (type instanceof FieldType.Int64FieldType) {
            return "Long";
        }
        if (type.isString()) {
            return "String";
        }
        if (type.isStruct()) {
            return type.toString();
        }
        throw new RuntimeException("Unsupported field type " + type);
    }

    private String readFieldFromStruct(FieldType type, String name) {
        if (type instanceof FieldType.BoolFieldType) {
            return String.format("struct.getBoolean(\"%s\")", name);
        }
        if (type instanceof FieldType.Int8FieldType) {
            return String.format("struct.getByte(\"%s\")", name);
        }
        if (type instanceof FieldType.Int16FieldType) {
            return String.format("struct.getShort(\"%s\")", name);
        }
        if (type instanceof FieldType.Int32FieldType) {
            return String.format("struct.getInt(\"%s\")", name);
        }
        if (type instanceof FieldType.Int64FieldType) {
            return String.format("struct.getLong(\"%s\")", name);
        }
        if (type.isString()) {
            return String.format("struct.getString(\"%s\")", name);
        }
        if (type.isBytes()) {
            return String.format("struct.getByteArray(\"%s\")", name);
        }
        if (type.isStruct()) {
            return String.format("new %s(struct, version)", type.toString());
        }
        throw new RuntimeException("Unsupported field type " + type);
    }

    private void generateClassWriter(String className, StructSpec struct, Versions parentVersions) {
        Versions curVersions;
        this.headerGenerator.addImport("org.apache.kafka.common.protocol.Writable");
        this.buffer.printf("@Override%n", new Object[0]);
        this.buffer.printf("public void write(Writable writable, short version) {%n", new Object[0]);
        this.buffer.incrementIndent();
        if (this.generateInverseVersionCheck(parentVersions, struct.versions())) {
            this.buffer.incrementIndent();
            this.headerGenerator.addImport("org.apache.kafka.common.errors.UnsupportedVersionException");
            this.buffer.printf("throw new UnsupportedVersionException(\"Can't write version \" + version + \" of %s\");%n", className);
            this.buffer.decrementIndent();
            this.buffer.printf("}%n", new Object[0]);
        }
        if ((curVersions = parentVersions.intersect(struct.versions())).empty()) {
            throw new RuntimeException("Version ranges " + parentVersions + " and " + struct.versions() + " have no versions in common.");
        }
        for (FieldSpec field : struct.fields()) {
            this.generateFieldWriter(field, curVersions);
        }
        this.buffer.decrementIndent();
        this.buffer.printf("}%n", new Object[0]);
    }

    private String writeFieldToWritable(FieldType type, boolean nullable, String name) {
        if (type instanceof FieldType.BoolFieldType) {
            return String.format("writable.writeByte(%s ? (byte) 1 : (byte) 0)", name);
        }
        if (type instanceof FieldType.Int8FieldType) {
            return String.format("writable.writeByte(%s)", name);
        }
        if (type instanceof FieldType.Int16FieldType) {
            return String.format("writable.writeShort(%s)", name);
        }
        if (type instanceof FieldType.Int32FieldType) {
            return String.format("writable.writeInt(%s)", name);
        }
        if (type instanceof FieldType.Int64FieldType) {
            return String.format("writable.writeLong(%s)", name);
        }
        if (type instanceof FieldType.StringFieldType) {
            if (nullable) {
                return String.format("writable.writeNullableString(%s)", name);
            }
            return String.format("writable.writeString(%s)", name);
        }
        if (type instanceof FieldType.BytesFieldType) {
            if (nullable) {
                return String.format("writable.writeNullableBytes(%s)", name);
            }
            return String.format("writable.writeBytes(%s)", name);
        }
        if (type instanceof FieldType.StructType) {
            return String.format("%s.write(writable, version)", name);
        }
        throw new RuntimeException("Unsupported field type " + type);
    }

    private void generateFieldWriter(FieldSpec field, Versions curVersions) {
        if (field.type().isArray()) {
            boolean maybeAbsent = this.generateVersionCheck(curVersions, field.versions());
            boolean maybeNull = this.generateNullCheck(curVersions, field);
            if (maybeNull) {
                this.buffer.printf("writable.writeInt(-1);%n", new Object[0]);
                this.buffer.decrementIndent();
                this.buffer.printf("} else {%n", new Object[0]);
                this.buffer.incrementIndent();
            }
            this.buffer.printf("writable.writeInt(%s.size());%n", field.camelCaseName());
            FieldType.ArrayType arrayType = (FieldType.ArrayType)field.type();
            FieldType elementType = arrayType.elementType();
            String nestedTypeName = elementType.isStruct() ? elementType.toString() : this.getBoxedJavaType(elementType);
            this.buffer.printf("for (%s element : %s) {%n", nestedTypeName, field.camelCaseName());
            this.buffer.incrementIndent();
            this.buffer.printf("%s;%n", this.writeFieldToWritable(elementType, false, "element"));
            this.buffer.decrementIndent();
            this.buffer.printf("}%n", new Object[0]);
            if (maybeNull) {
                this.buffer.decrementIndent();
                this.buffer.printf("}%n", new Object[0]);
            }
            if (maybeAbsent) {
                this.buffer.decrementIndent();
                this.buffer.printf("}%n", new Object[0]);
            }
        } else {
            boolean maybeAbsent = this.generateVersionCheck(curVersions, field.versions());
            this.buffer.printf("%s;%n", this.writeFieldToWritable(field.type(), !field.nullableVersions().empty(), field.camelCaseName()));
            if (maybeAbsent) {
                this.buffer.decrementIndent();
                this.buffer.printf("}%n", new Object[0]);
            }
        }
    }

    private void generateClassToStruct(String className, StructSpec struct, Versions parentVersions) {
        Versions curVersions;
        this.headerGenerator.addImport("org.apache.kafka.common.protocol.types.Struct");
        this.buffer.printf("@Override%n", new Object[0]);
        this.buffer.printf("public Struct toStruct(short version) {%n", new Object[0]);
        this.buffer.incrementIndent();
        if (this.generateInverseVersionCheck(parentVersions, struct.versions())) {
            this.buffer.incrementIndent();
            this.headerGenerator.addImport("org.apache.kafka.common.errors.UnsupportedVersionException");
            this.buffer.printf("throw new UnsupportedVersionException(\"Can't write version \" + version + \" of %s\");%n", className);
            this.buffer.decrementIndent();
            this.buffer.printf("}%n", new Object[0]);
        }
        if ((curVersions = parentVersions.intersect(struct.versions())).empty()) {
            throw new RuntimeException("Version ranges " + parentVersions + " and " + struct.versions() + " have no versions in common.");
        }
        this.buffer.printf("Struct struct = new Struct(SCHEMAS[version]);%n", new Object[0]);
        for (FieldSpec field : struct.fields()) {
            this.generateFieldToStruct(field, curVersions);
        }
        this.buffer.printf("return struct;%n", new Object[0]);
        this.buffer.decrementIndent();
        this.buffer.printf("}%n", new Object[0]);
    }

    private void generateFieldToStruct(FieldSpec field, Versions curVersions) {
        if (!field.type().canBeNullable() && !field.nullableVersions().empty()) {
            throw new RuntimeException("Fields of type " + field.type() + " cannot be nullable.");
        }
        if (field.type() instanceof FieldType.BoolFieldType || field.type() instanceof FieldType.Int8FieldType || field.type() instanceof FieldType.Int16FieldType || field.type() instanceof FieldType.Int32FieldType || field.type() instanceof FieldType.Int64FieldType || field.type() instanceof FieldType.StringFieldType) {
            boolean maybeAbsent = this.generateVersionCheck(curVersions, field.versions());
            this.buffer.printf("struct.set(\"%s\", this.%s);%n", field.snakeCaseName(), field.camelCaseName());
            if (maybeAbsent) {
                this.buffer.decrementIndent();
                this.buffer.printf("}%n", new Object[0]);
            }
        } else if (field.type().isBytes()) {
            boolean maybeAbsent = this.generateVersionCheck(curVersions, field.versions());
            this.buffer.printf("struct.setByteArray(\"%s\", this.%s);%n", field.snakeCaseName(), field.camelCaseName());
            if (maybeAbsent) {
                this.buffer.decrementIndent();
                this.buffer.printf("}%n", new Object[0]);
            }
        } else if (field.type().isArray()) {
            FieldType.ArrayType arrayType;
            FieldType elementType;
            boolean maybeNull;
            boolean maybeAbsent = this.generateVersionCheck(curVersions, field.versions());
            if (!maybeAbsent) {
                this.buffer.printf("{%n", new Object[0]);
                this.buffer.incrementIndent();
            }
            if (maybeNull = this.generateNullCheck(curVersions, field)) {
                this.buffer.printf("struct.set(\"%s\", null);%n", field.snakeCaseName());
                this.buffer.decrementIndent();
                this.buffer.printf("} else {%n", new Object[0]);
                this.buffer.incrementIndent();
            }
            String boxdElementType = (elementType = (arrayType = (FieldType.ArrayType)field.type()).elementType()).isStruct() ? "Struct" : this.getBoxedJavaType(elementType);
            this.buffer.printf("%s[] nestedObjects = new %s[%s.size()];%n", boxdElementType, boxdElementType, field.camelCaseName());
            this.buffer.printf("int i = 0;%n", new Object[0]);
            this.buffer.printf("for (%s element : this.%s) {%n", this.getBoxedJavaType(arrayType.elementType()), field.camelCaseName());
            this.buffer.incrementIndent();
            if (elementType.isStruct()) {
                this.buffer.printf("nestedObjects[i++] = element.toStruct(version);%n", new Object[0]);
            } else {
                this.buffer.printf("nestedObjects[i++] = element;%n", new Object[0]);
            }
            this.buffer.decrementIndent();
            this.buffer.printf("}%n", new Object[0]);
            this.buffer.printf("struct.set(\"%s\", (Object[]) nestedObjects);%n", field.snakeCaseName());
            if (maybeNull) {
                this.buffer.decrementIndent();
                this.buffer.printf("}%n", new Object[0]);
            }
            this.buffer.decrementIndent();
            this.buffer.printf("}%n", new Object[0]);
        } else {
            throw new RuntimeException("Unsupported field type " + field.type());
        }
    }

    private void generateClassSize(String className, StructSpec struct, Versions parentVersions) {
        Versions curVersions;
        this.buffer.printf("@Override%n", new Object[0]);
        this.buffer.printf("public int size(short version) {%n", new Object[0]);
        this.buffer.incrementIndent();
        this.buffer.printf("int size = 0;%n", new Object[0]);
        if (this.generateInverseVersionCheck(parentVersions, struct.versions())) {
            this.buffer.incrementIndent();
            this.headerGenerator.addImport("org.apache.kafka.common.errors.UnsupportedVersionException");
            this.buffer.printf("throw new UnsupportedVersionException(\"Can't size version \" + version + \" of %s\");%n", className);
            this.buffer.decrementIndent();
            this.buffer.printf("}%n", new Object[0]);
        }
        if ((curVersions = parentVersions.intersect(struct.versions())).empty()) {
            throw new RuntimeException("Version ranges " + parentVersions + " and " + struct.versions() + " have no versions in common.");
        }
        for (FieldSpec field : struct.fields()) {
            this.generateFieldSize(field, curVersions);
        }
        this.buffer.printf("return size;%n", new Object[0]);
        this.buffer.decrementIndent();
        this.buffer.printf("}%n", new Object[0]);
    }

    private void generateVariableLengthFieldSize(String fieldName, FieldType type, boolean nullable) {
        if (type instanceof FieldType.StringFieldType) {
            this.buffer.printf("size += 2;%n", new Object[0]);
            if (nullable) {
                this.buffer.printf("if (%s != null) {%n", fieldName);
                this.buffer.incrementIndent();
            }
            this.headerGenerator.addImport("org.apache.kafka.common.protocol.MessageUtil");
            this.buffer.printf("size += MessageUtil.serializedUtf8Length(%s);%n", fieldName);
            if (nullable) {
                this.buffer.decrementIndent();
                this.buffer.printf("}%n", new Object[0]);
            }
        } else if (type instanceof FieldType.BytesFieldType) {
            this.buffer.printf("size += 4;%n", new Object[0]);
            if (nullable) {
                this.buffer.printf("if (%s != null) {%n", fieldName);
                this.buffer.incrementIndent();
            }
            this.buffer.printf("size += %s.length;%n", fieldName);
            if (nullable) {
                this.buffer.decrementIndent();
                this.buffer.printf("}%n", new Object[0]);
            }
        } else if (type instanceof FieldType.StructType) {
            this.buffer.printf("size += %s.size(version);%n", fieldName);
        } else {
            throw new RuntimeException("Unsupported type " + type);
        }
    }

    private void generateFieldSize(FieldSpec field, Versions curVersions) {
        if (field.type().fixedLength().isPresent()) {
            boolean maybeAbsent = this.generateVersionCheck(curVersions, field.versions());
            this.buffer.printf("size += %d;%n", field.type().fixedLength().get());
            if (maybeAbsent) {
                this.buffer.decrementIndent();
                this.generateAbsentValueCheck(field);
            }
        } else if (field.type().isString() || field.type().isBytes() || field.type().isStruct()) {
            boolean nullable = !curVersions.intersect(field.nullableVersions()).empty();
            boolean maybeAbsent = this.generateVersionCheck(curVersions, field.versions());
            this.generateVariableLengthFieldSize(field.camelCaseName(), field.type(), nullable);
            if (maybeAbsent) {
                this.buffer.decrementIndent();
                this.generateAbsentValueCheck(field);
            }
        } else if (field.type().isArray()) {
            boolean maybeAbsent = this.generateVersionCheck(curVersions, field.versions());
            boolean maybeNull = this.generateNullCheck(curVersions, field);
            if (maybeNull) {
                this.buffer.printf("size += 4;%n", new Object[0]);
                this.buffer.decrementIndent();
                this.buffer.printf("} else {%n", new Object[0]);
                this.buffer.incrementIndent();
            }
            this.buffer.printf("size += 4;%n", new Object[0]);
            FieldType.ArrayType arrayType = (FieldType.ArrayType)field.type();
            FieldType elementType = arrayType.elementType();
            if (elementType.fixedLength().isPresent()) {
                this.buffer.printf("size += %s.size() * %d;%n", field.camelCaseName(), elementType.fixedLength().get());
            } else {
                if (elementType instanceof FieldType.ArrayType) {
                    throw new RuntimeException("Arrays of arrays are not supported (use a struct).");
                }
                this.buffer.printf("for (%s element : %s) {%n", this.getBoxedJavaType(elementType), field.camelCaseName());
                this.buffer.incrementIndent();
                this.generateVariableLengthFieldSize("element", elementType, false);
                this.buffer.decrementIndent();
                this.buffer.printf("}%n", new Object[0]);
            }
            if (maybeNull) {
                this.buffer.decrementIndent();
                this.buffer.printf("}%n", new Object[0]);
            }
            if (maybeAbsent) {
                this.buffer.decrementIndent();
                this.generateAbsentValueCheck(field);
            }
        } else {
            throw new RuntimeException("Unsupported field type " + field.type());
        }
    }

    private void generateAbsentValueCheck(FieldSpec field) {
        if (field.ignorable()) {
            this.buffer.printf("}%n", new Object[0]);
            return;
        }
        this.buffer.printf("} else {%n", new Object[0]);
        this.buffer.incrementIndent();
        this.headerGenerator.addImport("org.apache.kafka.common.errors.UnsupportedVersionException");
        if (field.type().isArray()) {
            this.buffer.printf("if (!%s.isEmpty()) {%n", field.camelCaseName());
        } else if (field.type().isBytes()) {
            this.buffer.printf("if (%s.length != 0) {%n", field.camelCaseName());
        } else if (field.type().isString()) {
            if (this.fieldDefault(field).equals("null")) {
                this.buffer.printf("if (%s != null) {%n", field.camelCaseName());
            } else {
                this.buffer.printf("if (!%s.equals(%s)) {%n", field.camelCaseName(), this.fieldDefault(field));
            }
        } else if (field.type() instanceof FieldType.BoolFieldType) {
            this.buffer.printf("if (%s%s) {%n", this.fieldDefault(field).equals("true") ? "!" : "", field.camelCaseName());
        } else {
            this.buffer.printf("if (%s != %s) {%n", field.camelCaseName(), this.fieldDefault(field));
        }
        this.buffer.incrementIndent();
        this.buffer.printf("throw new UnsupportedVersionException(\"Attempted to write a non-default %s at version \" + version);%n", field.camelCaseName());
        this.buffer.decrementIndent();
        this.buffer.printf("}%n", new Object[0]);
        this.buffer.decrementIndent();
        this.buffer.printf("}%n", new Object[0]);
    }

    private void generateClassEquals(String className, StructSpec struct, boolean onlyMapKeys) {
        this.buffer.printf("@Override%n", new Object[0]);
        this.buffer.printf("public boolean equals(Object obj) {%n", new Object[0]);
        this.buffer.incrementIndent();
        this.buffer.printf("if (!(obj instanceof %s)) return false;%n", className);
        if (!struct.fields().isEmpty()) {
            this.buffer.printf("%s other = (%s) obj;%n", className, className);
            for (FieldSpec field : struct.fields()) {
                if (onlyMapKeys && !field.mapKey()) continue;
                this.generateFieldEquals(field);
            }
        }
        this.buffer.printf("return true;%n", new Object[0]);
        this.buffer.decrementIndent();
        this.buffer.printf("}%n", new Object[0]);
    }

    private void generateFieldEquals(FieldSpec field) {
        if (field.type().isString() || field.type().isArray() || field.type().isStruct()) {
            this.buffer.printf("if (this.%s == null) {%n", field.camelCaseName());
            this.buffer.incrementIndent();
            this.buffer.printf("if (other.%s != null) return false;%n", field.camelCaseName());
            this.buffer.decrementIndent();
            this.buffer.printf("} else {%n", new Object[0]);
            this.buffer.incrementIndent();
            this.buffer.printf("if (!this.%s.equals(other.%s)) return false;%n", field.camelCaseName(), field.camelCaseName());
            this.buffer.decrementIndent();
            this.buffer.printf("}%n", new Object[0]);
        } else if (field.type().isBytes()) {
            this.headerGenerator.addImport("java.util.Arrays");
            this.buffer.printf("if (!Arrays.equals(this.%s, other.%s)) return false;%n", field.camelCaseName(), field.camelCaseName());
        } else {
            this.buffer.printf("if (%s != other.%s) return false;%n", field.camelCaseName(), field.camelCaseName());
        }
    }

    private void generateClassHashCode(StructSpec struct, boolean onlyMapKeys) {
        this.buffer.printf("@Override%n", new Object[0]);
        this.buffer.printf("public int hashCode() {%n", new Object[0]);
        this.buffer.incrementIndent();
        this.buffer.printf("int hashCode = 0;%n", new Object[0]);
        for (FieldSpec field : struct.fields()) {
            if (onlyMapKeys && !field.mapKey()) continue;
            this.generateFieldHashCode(field);
        }
        this.buffer.printf("return hashCode;%n", new Object[0]);
        this.buffer.decrementIndent();
        this.buffer.printf("}%n", new Object[0]);
    }

    private void generateFieldHashCode(FieldSpec field) {
        if (field.type() instanceof FieldType.BoolFieldType) {
            this.buffer.printf("hashCode = 31 * hashCode + (%s ? 1231 : 1237);%n", field.camelCaseName());
        } else if (field.type() instanceof FieldType.Int8FieldType || field.type() instanceof FieldType.Int16FieldType || field.type() instanceof FieldType.Int32FieldType) {
            this.buffer.printf("hashCode = 31 * hashCode + %s;%n", field.camelCaseName());
        } else if (field.type() instanceof FieldType.Int64FieldType) {
            this.buffer.printf("hashCode = 31 * hashCode + ((int) (%s >> 32) ^ (int) %s);%n", field.camelCaseName(), field.camelCaseName());
        } else if (field.type().isString()) {
            this.buffer.printf("hashCode = 31 * hashCode + (%s == null ? 0 : %s.hashCode());%n", field.camelCaseName(), field.camelCaseName());
        } else if (field.type().isBytes()) {
            this.headerGenerator.addImport("java.util.Arrays");
            this.buffer.printf("hashCode = 31 * hashCode + Arrays.hashCode(%s);%n", field.camelCaseName());
        } else if (field.type().isStruct() || field.type().isArray()) {
            this.buffer.printf("hashCode = 31 * hashCode + (%s == null ? 0 : %s.hashCode());%n", field.camelCaseName(), field.camelCaseName());
        } else {
            throw new RuntimeException("Unsupported field type " + field.type());
        }
    }

    private void generateClassToString(String className, StructSpec struct) {
        this.buffer.printf("@Override%n", new Object[0]);
        this.buffer.printf("public String toString() {%n", new Object[0]);
        this.buffer.incrementIndent();
        this.buffer.printf("return \"%s(\"%n", className);
        this.buffer.incrementIndent();
        String prefix = "";
        for (FieldSpec field : struct.fields()) {
            this.generateFieldToString(prefix, field);
            prefix = ", ";
        }
        this.buffer.printf("+ \")\";%n", new Object[0]);
        this.buffer.decrementIndent();
        this.buffer.decrementIndent();
        this.buffer.printf("}%n", new Object[0]);
    }

    private void generateFieldToString(String prefix, FieldSpec field) {
        if (field.type() instanceof FieldType.BoolFieldType) {
            this.buffer.printf("+ \"%s%s=\" + (%s ? \"true\" : \"false\")%n", prefix, field.camelCaseName(), field.camelCaseName());
        } else if (field.type() instanceof FieldType.Int8FieldType || field.type() instanceof FieldType.Int16FieldType || field.type() instanceof FieldType.Int32FieldType || field.type() instanceof FieldType.Int64FieldType) {
            this.buffer.printf("+ \"%s%s=\" + %s%n", prefix, field.camelCaseName(), field.camelCaseName());
        } else if (field.type().isString()) {
            this.buffer.printf("+ \"%s%s='\" + %s + \"'\"%n", prefix, field.camelCaseName(), field.camelCaseName());
        } else if (field.type().isBytes()) {
            this.headerGenerator.addImport("java.util.Arrays");
            this.buffer.printf("+ \"%s%s=\" + Arrays.toString(%s)%n", prefix, field.camelCaseName(), field.camelCaseName());
        } else if (field.type().isStruct()) {
            this.buffer.printf("+ \"%s%s=\" + %s.toString()%n", prefix, field.camelCaseName(), field.camelCaseName());
        } else if (field.type().isArray()) {
            this.headerGenerator.addImport("org.apache.kafka.common.protocol.MessageUtil");
            if (field.nullableVersions().empty()) {
                this.buffer.printf("+ \"%s%s=\" + MessageUtil.deepToString(%s.iterator())%n", prefix, field.camelCaseName(), field.camelCaseName());
            } else {
                this.buffer.printf("+ \"%s%s=\" + ((%s == null) ? \"null\" : MessageUtil.deepToString(%s.iterator()))%n", prefix, field.camelCaseName(), field.camelCaseName(), field.camelCaseName());
            }
        } else {
            throw new RuntimeException("Unsupported field type " + field.type());
        }
    }

    private boolean generateNullCheck(Versions prevVersions, FieldSpec field) {
        if (prevVersions.intersect(field.nullableVersions()).empty()) {
            return false;
        }
        this.buffer.printf("if (%s == null) {%n", field.camelCaseName());
        this.buffer.incrementIndent();
        return true;
    }

    private boolean generateVersionCheck(Versions prev, Versions cur) {
        if (cur.lowest() > prev.lowest()) {
            if (cur.highest() < prev.highest()) {
                this.buffer.printf("if ((version >= %d) && (version <= %d)) {%n", cur.lowest(), cur.highest());
                this.buffer.incrementIndent();
                return true;
            }
            this.buffer.printf("if (version >= %d) {%n", cur.lowest());
            this.buffer.incrementIndent();
            return true;
        }
        if (cur.highest() < prev.highest()) {
            this.buffer.printf("if (version <= %d) {%n", cur.highest());
            this.buffer.incrementIndent();
            return true;
        }
        return false;
    }

    private boolean generateInverseVersionCheck(Versions prev, Versions cur) {
        if (cur.lowest() > prev.lowest()) {
            if (cur.highest() < prev.highest()) {
                this.buffer.printf("if ((version < %d) || (version > %d)) {%n", cur.lowest(), cur.highest());
                return true;
            }
            this.buffer.printf("if (version < %d) {%n", cur.lowest());
            return true;
        }
        if (cur.highest() < prev.highest()) {
            this.buffer.printf("if (version > %d) {%n", cur.highest());
            return true;
        }
        return false;
    }

    private void generateSetDefault(FieldSpec field) {
        this.buffer.decrementIndent();
        this.buffer.printf("} else {%n", new Object[0]);
        this.buffer.incrementIndent();
        this.buffer.printf("this.%s = %s;%n", field.camelCaseName(), this.fieldDefault(field));
        this.buffer.decrementIndent();
        this.buffer.printf("}%n", new Object[0]);
    }

    private String fieldDefault(FieldSpec field) {
        if (field.type() instanceof FieldType.BoolFieldType) {
            if (field.defaultString().isEmpty()) {
                return "false";
            }
            if (field.defaultString().equalsIgnoreCase("true")) {
                return "true";
            }
            if (field.defaultString().equalsIgnoreCase("false")) {
                return "false";
            }
            throw new RuntimeException("Invalid default for boolean field " + field.name() + ": " + field.defaultString());
        }
        if (field.type() instanceof FieldType.Int8FieldType) {
            if (field.defaultString().isEmpty()) {
                return "(byte) 0";
            }
            try {
                Byte.decode(field.defaultString());
            }
            catch (NumberFormatException e) {
                throw new RuntimeException("Invalid default for int8 field " + field.name() + ": " + field.defaultString(), e);
            }
            return "(byte) " + field.defaultString();
        }
        if (field.type() instanceof FieldType.Int16FieldType) {
            if (field.defaultString().isEmpty()) {
                return "(short) 0";
            }
            try {
                Short.decode(field.defaultString());
            }
            catch (NumberFormatException e) {
                throw new RuntimeException("Invalid default for int16 field " + field.name() + ": " + field.defaultString(), e);
            }
            return "(short) " + field.defaultString();
        }
        if (field.type() instanceof FieldType.Int32FieldType) {
            if (field.defaultString().isEmpty()) {
                return "0";
            }
            try {
                Integer.decode(field.defaultString());
            }
            catch (NumberFormatException e) {
                throw new RuntimeException("Invalid default for int32 field " + field.name() + ": " + field.defaultString(), e);
            }
            return field.defaultString();
        }
        if (field.type() instanceof FieldType.Int64FieldType) {
            if (field.defaultString().isEmpty()) {
                return "0L";
            }
            try {
                Integer.decode(field.defaultString());
            }
            catch (NumberFormatException e) {
                throw new RuntimeException("Invalid default for int64 field " + field.name() + ": " + field.defaultString(), e);
            }
            return field.defaultString() + "L";
        }
        if (field.type() instanceof FieldType.StringFieldType) {
            if (field.defaultString().equals("null")) {
                if (!field.nullableVersions().contains(field.versions())) {
                    throw new RuntimeException("null cannot be the default for field " + field.name() + ", because not all versions of this field are nullable.");
                }
                return "null";
            }
            return "\"" + field.defaultString() + "\"";
        }
        if (field.type().isBytes()) {
            if (!field.defaultString().isEmpty()) {
                throw new RuntimeException("Invalid default for bytes field " + field.name() + ": custom defaults are not supported for bytes fields.");
            }
            this.headerGenerator.addImport("org.apache.kafka.common.utils.Bytes");
            return "Bytes.EMPTY";
        }
        if (field.type().isStruct()) {
            if (!field.defaultString().isEmpty()) {
                throw new RuntimeException("Invalid default for struct field " + field.name() + ": custom defaults are not supported for struct fields.");
            }
            return "new " + field.type().toString() + "()";
        }
        if (field.type().isArray()) {
            if (!field.defaultString().isEmpty()) {
                throw new RuntimeException("Invalid default for array field " + field.name() + ": custom defaults are not supported for array fields.");
            }
            FieldType.ArrayType arrayType = (FieldType.ArrayType)field.type();
            if (field.toStruct().hasKeys()) {
                return "new " + MessageDataGenerator.collectionType(arrayType.elementType().toString()) + "(0)";
            }
            this.headerGenerator.addImport("java.util.ArrayList");
            return "new ArrayList<" + this.getBoxedJavaType(arrayType.elementType()) + ">()";
        }
        throw new RuntimeException("Unsupported field type " + field.type());
    }

    private void generateFieldAccessor(FieldSpec field) {
        this.buffer.printf("%n", new Object[0]);
        this.generateAccessor(this.fieldAbstractJavaType(field), field.camelCaseName(), field.camelCaseName());
    }

    private void generateAccessor(String javaType, String functionName, String memberName) {
        this.buffer.printf("public %s %s() {%n", javaType, functionName);
        this.buffer.incrementIndent();
        this.buffer.printf("return this.%s;%n", memberName);
        this.buffer.decrementIndent();
        this.buffer.printf("}%n", new Object[0]);
    }

    private void generateFieldMutator(String className, FieldSpec field) {
        this.buffer.printf("%n", new Object[0]);
        this.buffer.printf("public %s set%s(%s v) {%n", className, field.capitalizedCamelCaseName(), this.fieldAbstractJavaType(field));
        this.buffer.incrementIndent();
        this.buffer.printf("this.%s = v;%n", field.camelCaseName());
        this.buffer.printf("return this;%n", new Object[0]);
        this.buffer.decrementIndent();
        this.buffer.printf("}%n", new Object[0]);
    }

    private void generateSetter(String javaType, String functionName, String memberName) {
        this.buffer.printf("public void %s(%s v) {%n", functionName, javaType);
        this.buffer.incrementIndent();
        this.buffer.printf("this.%s = v;%n", memberName);
        this.buffer.decrementIndent();
        this.buffer.printf("}%n", new Object[0]);
    }
}

