/*
 *  Copyright 2002-2022 Barcelona Supercomputing Center (www.bsc.es)
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 */

#ifndef _GNU_SOURCE
#define _GNU_SOURCE 1
#endif

#include <stdlib.h>
#include <limits.h>
#include <string.h>
#include <ctype.h>
#include "backend.h"
#include "semantic.h"
#include "backendlib.h"
#include "types.h"

#define BUFF_ELEMENTS 9
#define ARGS_OFFSET 5
typedef int bool;
enum { false, true };

static FILE *stubsFile = NULL;
static FILE *workerFile = NULL;
static FILE *includeFile = NULL;

static char includeName[PATH_MAX];
static char *c_types[] = {
    "int",			        // boolean_dt
    "char",			        // char_dt
    "unsigned char",	    // byte_dt
    "short",			    // short_dt
    "int",			        // int_dt
    "long",			        // long_dt
    "float",			    // float_dt
    "double",			    // double_dt
    "char *",			    // string_dt
    "char *",			    // string_64_dt
    "file",			        // file_dt
    "void *",			    // object_dt
    "void *",			    // psco_dt
    "void *",			    // external_psco_dt
    "void *",			    // binding_object_dt
    "char", 			    // wchar_dt
    "char *", 		        // wstring_dt
    "long long", 		    // longlong_dt
    "void", 			    // void_dt
    "void",			        // any_dt
    "char",	  	  	        //array_char_dt,
    "unsigned char",        //array_byte_dt,
    "short",		        //array_short_dt,
    "int",			        //array_int_dt,
    "long", 		        //array_long_dt,
    "float",		        //array_float_dt,
    "double", 		        //array_double_dt,
    "collection",           //collection_t
    "stream",               //stream_t
    "external_stream_t",    //binding streams
    "enum",                 //enum_dt
    "error"  		        // null_dt
};

static char *c_out_types[] = {
    "int",			        // boolean_dt
    "char",			        // char_dt
    "unsigned char",	    // byte_dt
    "short",			    // short_dt
    "int",			        // int_dt
    "long",			        // long_dt
    "float",			    // float_dt
    "double",			    // double_dt
    "char *",			    // string_dt
    "char *",			    // string_64_dt
    "file",			        // file_dt
    "void *",			    // object_dt
    "void *",			    // psco_dt
    "void *",			    // external_psco_dt
    "void *",			    // binding_object_dt
    "char", 			    // wchar_dt
    "char *", 		        // wstring_dt
    "long long", 		    // longlong_dt
    "void", 			    // void_dt
    "void",			        // any_dt
    "char",	  	  	        //array_char_dt,
    "unsigned char",        //array_byte_dt,
    "short",		        //array_short_dt,
    "int",			        //array_int_dt,
    "long", 		        //array_long_dt,
    "float",		        //array_float_dt,
    "double", 		        //array_double_dt,
    "collection",           //collection_t
    "stream",               //stream_t
    "external_stream_t",    //binding streams
    "enum",                 //enum_dt
    "error"  		        // null_dt
};

void asprintf_error(char* pointer, char* error) {
    free(pointer);
    printf(error);
    exit(1);
}

char* concat(const char *s1, const char *s2) {
    const size_t len1 = strlen(s1);
    const size_t len2 = strlen(s2);
    char *result = malloc(len1+len2+1);//+1 for the null-terminator
    //in real code you would check for errors in malloc here
    memcpy(result, s1, len1);
    memcpy(result+len1, s2, len2+1);//+1 to copy the null-terminator
    return result;
}

/*
 * Add headers and includes for the master stubs file
 */
void generate_stubs_prolog() {
    fprintf(stubsFile, "/* This file has been autogenerated from '%s'. */\n", get_filename());
    fprintf(stubsFile, "/* CHANGES TO THIS FILE WILL BE LOST */\n");
    fprintf(stubsFile, "\n");
    fprintf(stubsFile, "#include <stdio.h>\n");
    fprintf(stubsFile, "#include <stdlib.h>\n");
    fprintf(stubsFile, "#include <limits.h>\n");
    fprintf(stubsFile, "#include <string.h>\n");
    fprintf(stubsFile, "#include <fstream>\n");
    fprintf(stubsFile, "#include <sstream>\n");
    fprintf(stubsFile, "#include <c_compss.h>\n");
    //fprintf(stubsFile, "#include <param_metadata.h>\n");
    fprintf(stubsFile, "#include \"%s\"\n", includeName);

    fprintf(stubsFile, "\n");
    fprintf(stubsFile, "using namespace std;\n");
    fprintf(stubsFile, "using namespace boost;\n");

    fprintf(stubsFile, "\n");
}

/*
 * Add headers and includes for the worker stubs file
 */
void generate_worker_prolog() {
    fprintf(workerFile, "/* This file has been autogenerated from '%s'. */\n", get_filename());
    fprintf(workerFile, "/* CHANGES TO THIS FILE WILL BE LOST */\n");
    fprintf(workerFile, "\n");
    fprintf(workerFile, "#include <string.h>\n");
    fprintf(workerFile, "#include <stdio.h>\n");
    fprintf(workerFile, "#include <stdlib.h>\n");
    fprintf(workerFile, "#include <limits.h>\n");
    fprintf(workerFile, "#include <string.h>\n");
    fprintf(workerFile, "#include <fstream>\n");
    fprintf(workerFile, "#include <sstream>\n");

    //fprintf(workerFile, "#include <compss_worker_lock.h>\n");
    fprintf(workerFile, "#include <c_compss_commons.h>\n");
    fprintf(workerFile, "#include \"%s\"\n", includeName);
    fprintf(workerFile, "\n");
    fprintf(workerFile, "using namespace std;\n");
    fprintf(workerFile, "\n");


    fprintf(workerFile, "#ifdef OMPSS2_ENABLED\n");
    fprintf(workerFile, "#include <pthread.h>\n");
    fprintf(workerFile, "#include <nanos6/bootstrap.h>\n");
    fprintf(workerFile, "#include <nanos6/library-mode.h>\n");
    fprintf(workerFile, "#endif\n");

}

/*
 * Generate the needed preprocessor conditions to intialize Nanos++ or Nanos6
 */
void generate_nanos_initialization() {

    //OmpSs2 Nanos6 initialization
    fprintf(workerFile, "#ifdef OMPSS2_ENABLED\n");
    fprintf(workerFile, "\t\t#ifdef OMPSS_ENABLED\n");
    fprintf(workerFile, "\t\t\t#error Only one of both OmpSs2 and OmpSs can be enabled.\n");
    fprintf(workerFile, "\t\t#endif\n");
    fprintf(workerFile, "#endif\n\n");

    fprintf(workerFile, "#ifdef OMPSS_ENABLED\n");
    fprintf(workerFile, "\t\t#ifdef OMPSS2_ENABLED\n");
    fprintf(workerFile, "\t\t\t#error Only one of both OmpSs2 and OmpSs can be enabled.\n");
    fprintf(workerFile, "\t\t#endif\n");
    fprintf(workerFile, " cout << \"DEBUG LOG: nanos ADMIT CURRENT THREAD\" << endl;\n");
    fprintf(workerFile, "\t\t nanos_admit_current_thread();\n");
    fprintf(workerFile, "#endif\n");
    fprintf(workerFile, "}\n");

}

/*
 * Generate the needed preprocessor conditions to shutdown Nanos++ or Nanos6
 */
void generate_nanos_shutdown() {

    //OmpSs Nanos++ shutdown
    fprintf(workerFile, "#ifdef OMPSS_ENABLED\n");
    fprintf(workerFile, "cout << \"DEBUG LOG: nanos LEAVE TEAM CURRENT THREAD\" << endl;\n");
    fprintf(workerFile, "\t\t nanos_leave_team();\n");
    fprintf(workerFile, "\t\t nanos_expel_current_thread();\n");
    fprintf(workerFile, "#endif\n");
    fprintf(workerFile, "}\n");

}

/*
 * This function creates a struct with pthread_cond_var to
 * ease the use of condition variables.
 */
static void generate_cond_var_struct(FILE *outFile) {

    fprintf(outFile, "typedef struct {\n");
    fprintf(outFile, "\tpthread_mutex_t mutex;\n");
    fprintf(outFile, "\tpthread_cond_t cond;\n");
    fprintf(outFile, "\tint signaled;\n");
    fprintf(outFile, "} condition_variable_t;\n\n");

}

/*
 * This function generates the code associated with the callback
 * that Nanos6 will call to signal the worker because the OmpSs-2
 * task has finished.
 */
static void generate_cond_var_auxiliar_functions(FILE *outFile) {
    fprintf(outFile, "void condition_variable_callback(void *args) {\n");
    fprintf(outFile, "\tcondition_variable_t *cond_var = (condition_variable_t *) args;\n");
    fprintf(outFile, "\tpthread_mutex_lock(&cond_var->mutex);\n");
    fprintf(outFile, "\tcond_var->signaled = 1;\n");
    fprintf(outFile, "\tpthread_cond_signal(&cond_var->cond);\n");
    fprintf(outFile, "\tpthread_mutex_unlock(&cond_var->mutex);\n");
    fprintf(outFile, "}\n\n");

    fprintf(outFile, "void wait_condition_variable(condition_variable_t *cond_var) {\n");
    fprintf(outFile, "\tpthread_mutex_lock(&cond_var->mutex);\n");
    fprintf(outFile, "\twhile (cond_var->signaled == 0) {\n");
    fprintf(outFile, "\t\tpthread_cond_wait(&cond_var->cond, &cond_var->mutex);\n");
    fprintf(outFile, "\t}\n");
    fprintf(outFile, "pthread_mutex_unlock(&cond_var->mutex);\n");
    fprintf(outFile, "}\n\n");
}

/*
 * This function takes a function and generates a code that assign the actual
 * arguments and to the variables of the struct. This is useful for OmpSs-2
 * integration in terms of copying the values to spawn the wrapper function.
 *
 * It assumes that the struct name is of the form %s_struct where %s is the
 * name of the method to be called.
 *
 */
static void assign_arguments_to_struct(FILE *outFile, function* func) {

    char* struct_name;
    int printed_chars = 0;

    printed_chars = asprintf(&struct_name, "%s_struct", func->methodname);

    if (printed_chars < 0) {
        asprintf_error(struct_name, "Not possible to create the struct name.\n");
    }

    argument* args = func->first_argument;
    while (args != NULL) {
        fprintf(outFile, "\t\t\t%s.%s = %s;\n", struct_name, args->name, args->name);
        args = args->next_argument;
    }

}

/*
 * This function takes a function and generates a code that assigns the
 * values inside the struct (also the return?) to the arguments of the
 * function. This is useful for OmpSs-2 integration, to get the values
 * back to the variables
 *
 * It assumes that the struct name is of the form %s_struct where %s is the
 * name of the method to be called.
 */
static void assign_struct_to_arguments(FILE *outFile, function* func) {

    char* struct_name;
    int printed_chars = 0;

    printed_chars = asprintf(&struct_name, "struct_", func->methodname);

    if (printed_chars < 0) {
        asprintf_error(struct_name, "Not possible to create the struct name.\n");
    }

    argument* args = func->first_argument;
    while (args != NULL) {
        fprintf(outFile, "\t%s = %s.%s;\n", args->name, struct_name, args->name);
        args = args->next_argument;
    }

}

/*
 * This function takes a function and generates a wrapper to be executed
 * by Nanos6 as a task.
 */
static void generate_nanos6_wrapper(FILE *outFile, function* func) {
    fprintf(outFile, "void %s_wrapper(void* args) {\n", func->methodname);

    fprintf(outFile, "\t%s_struct_t* struct_ = (%s_struct_t*) args;\n", func->methodname, func->methodname);

    char* func_to_execute;
    int printed_chars = 0;

    if (func->return_type != void_dt && func->return_type != NULL) {
        fprintf(outFile, "\tstruct_->ret = ", func->return_type);
    }

    if ((func->classname != NULL) && (func->access_static == 0)) {
        printed_chars = asprintf(&func_to_execute, "\t%s->%s(", func->name, func->methodname);
    } else {
        printed_chars = asprintf(&func_to_execute, "\t%s(", func->name);
    }

    if (printed_chars < 0) {
        asprintf_error(func_to_execute, "ERROR: Not possible to generate method execution.\n");
    }

    fprintf(outFile, "%s", func_to_execute);

    argument* args = func->first_argument;

    int first = 1;
    while (args != NULL) {
        if (first) {
            first = 0;
        }
        else {
            fprintf(outFile, ", ");
        }
        fprintf(outFile, "struct_->%s", args->name);

        args = args->next_argument;
    }
    fprintf(outFile, ");\n");

    //assign_struct_to_arguments(outFile, func); //Assign struct values to variables

    fprintf(outFile, "}\n");
}

/*
 * Generate prototype for the initThread method at worker
 */
void generate_initThread_prototype(FILE *outFile) {
    fprintf(outFile, "void GE_initThread() {\n");
    fprintf(outFile, "\n");
}

void generate_worker_initThread() {
	generate_initThread_prototype(workerFile);
    generate_nanos_initialization();
}

void generate_finishThread_prototype(FILE *outFile) {
    fprintf(outFile, "void GE_finishThread() {\n");
    fprintf(outFile, "\n");
}

void generate_worker_finishThread() {
    generate_finishThread_prototype(workerFile);
    generate_nanos_shutdown();
}

void generate_initThread_empty(FILE *outFile) {
    generate_initThread_prototype(outFile);
	fprintf(outFile, "return;\n");
	fprintf(outFile, "}\n");
}

void generate_finishThread_empty(FILE *outFile) {
	generate_finishThread_prototype(outFile);
	fprintf(outFile, "return;\n");
	fprintf(outFile, "}\n");
}

/*
 * Generate prototype for the execute method at worker
 */
void generate_executor_prototype(FILE *outFile) {
    fprintf(outFile, "int execute(int argc, char **argv, CBindingCache* cache, int serializeOuts) {\n");
    fprintf(outFile, "\n");
}

/*
 * Starting of the headers file
 */
void generate_includes_prolog() {
    // Include file headers
    char *c;
    fprintf(includeFile, "/* This file must be #included in the actual implementation file. */\n");
    fprintf(includeFile, "/* This file has been autogenerated from '%s'. */\n", get_filename());
    fprintf(includeFile, "/* CHANGES TO THIS FILE WILL BE LOST */\n");
    fprintf(includeFile, "\n");
    fprintf(includeFile, "#ifndef _GSS_");
    for (c = includeName; *c; c++) {
        if (isalnum(*c)) {
            fprintf(includeFile, "%c", toupper(*c));
        } else {
            fprintf(includeFile, "_");
        }
    }
    fprintf(includeFile, "\n");

    fprintf(includeFile, "#define _GSS_");
    for (c = includeName; *c; c++) {
        if (isalnum(*c)) {
            fprintf(includeFile, "%c", toupper(*c));
        } else {
            fprintf(includeFile, "_");
        }
    }
    fprintf(includeFile, "\n");
    fprintf(includeFile, "#include <c_compss.h>\n");
    fprintf(includeFile, "\n");
    fprintf(includeFile, "typedef char* file;\n");
    fprintf(includeFile, "\n");

}

void generate_prolog() {
    char name[PATH_MAX];

    strncpy(name, get_filename_base(), PATH_MAX);
    strncat(name, "-stubs.cc", PATH_MAX);
    rename_if_clash(name);
    stubsFile = fopen(name, "w");
    if (stubsFile == NULL) {
        fprintf(stderr, "Error: Could not open %s for writing.\n", name);
        exit(1);
    }

    strncpy(name, get_filename_base(), PATH_MAX);
    strncat(name, "-executor.cc", PATH_MAX);
    rename_if_clash(name);
    workerFile = fopen(name, "w");
    if (workerFile == NULL) {
        fprintf(stderr, "Error: Could not open %s for writing.\n", name);
        exit(1);
    }

    strncpy(includeName, get_filename_base(), PATH_MAX);
    strncat(includeName, ".h", PATH_MAX);
    rename_if_clash(includeName);
    includeFile = fopen(includeName, "w");
    if (includeFile == NULL) {
        fprintf(stderr, "Error: Could not open %s for writing.\n", includeName);
        exit(1);
    }

    generate_stubs_prolog();

    generate_worker_prolog();

    generate_includes_prolog();

}

/*
 * Adding end part of the executor method at worker
 */
void generate_executor_end() {
    // Close switch clause
    fprintf(workerFile, "\t }\n");
    fprintf(workerFile, "\n");
    // If this point is reached, no operation has been selected
    // Raise error for incorrect method execution
    fprintf(workerFile, "\t cout << \"Incorrect Operation Code. Aborting...\"<< endl << flush;\n");
    fprintf(workerFile, "\t return -1;\n");
    fprintf(workerFile, "}\n");
}

void generate_epilogue(void) {
    char *c;
    fprintf(includeFile, "\n");
    fprintf(includeFile, "#endif /* _GSS_");
    for (c = includeName; *c; c++) {
        if (isalnum(*c)) {
            fprintf(includeFile, "%c", toupper(*c));
        } else {
            fprintf(includeFile, "_");
        }
    }

    fprintf(includeFile, " */\n");

    fclose(stubsFile);
    fclose(workerFile);
    fclose(includeFile);
}
/*
 * Generate task type code enum
 */
static void generate_enum(FILE *outFile, function *first_function) {
    function *func;
    int is_first = 1;
    int n = 0;

    fprintf(outFile, "enum operationCode {");

    func = first_function;
    while (func != NULL) {
        if (is_first) {
            is_first = 0;
        } else {
            fprintf(outFile, ", ");
        }
        char *func_name = strdup(func->name);
        replace_char(func_name, ':', '_');
        fprintf(outFile, "%sOp", func_name);
        n++;
        func = func->next_function;
    }

    fprintf(outFile, "};\n");

    is_first = 1;

    fprintf(outFile, "static const char *operationName[] = {");

    func = first_function;
    while (func != NULL) {
        if (is_first) {
            is_first = 0;
        } else {
            fprintf(outFile, ", ");
        }
        fprintf(outFile, "\"%s\"", func->name);
        func = func->next_function;
    }

    fprintf(outFile, "};\n");

    // Add constants (according to COMPSs Runtime)
    fprintf(outFile, "static const int N_OPS=%d;\n", n);
    fprintf(outFile, "static const int NUM_BASE_ARGS = 10;\n");
    fprintf(outFile, "static const int MIN_NUM_INTERNAL_ARGS = 15;\n");
    fprintf(outFile, "static const int METHOD_NAME_POS = 7;\n");
    fprintf(outFile,"\n");
}

/*
 * Return the string "type name" of the argument
 */
static char* construct_type_and_name(argument* arg) {

    char* ret;
    int printed_chars = -1;

    if (arg->dir == in_dir) {
        switch (arg->type) {
            case char_dt:
            case wchar_dt:
            case boolean_dt:
            case short_dt:
            case long_dt:
            case longlong_dt:
            case int_dt:
            case float_dt:
            case double_dt:
                printed_chars = asprintf(&ret, "%s %s", c_out_types[arg->type], arg->name);
                break;
            case object_dt:
                printed_chars = asprintf(&ret, "%s *%s", arg->classname, arg->name);
                break;
            case string_dt:
            case string_64_dt:
            case wstring_dt:
                printed_chars = asprintf(&ret, "%s %s", c_out_types[arg->type], arg->name);
                break;
            case file_dt:
                printed_chars = asprintf(&ret, "%s %s", c_out_types[arg->type], arg->name);
                break;
            case array_char_dt:
            case array_byte_dt:
            case array_short_dt:
            case array_int_dt:
            case array_long_dt:
            case array_float_dt:
            case array_double_dt:
                //TODO Check if * or & is needed
                printed_chars = asprintf(&ret, "%s *%s", c_out_types[arg->type], arg->name);
                break;
            case enum_dt:
                printed_chars = asprintf(&ret, "%s %s %s", c_out_types[arg->type], arg->classname, arg->name);
                break;
            case void_dt:
            case any_dt:
            case null_dt:
            default:
                ;
        }
    }
    else {
        switch (arg->type) {
            case char_dt:
            case wchar_dt:
            case boolean_dt:
            case short_dt:
            case long_dt:
            case longlong_dt:
            case int_dt:
            case float_dt:
            case double_dt:
                printed_chars = asprintf(&ret, "%s *%s", c_out_types[arg->type], arg->name);
                break;
            case object_dt:
                printed_chars = asprintf(&ret, "%s* %s", arg->classname, arg->name);
                break;
            case string_dt:
            case string_64_dt:
            case wstring_dt:
                printed_chars = asprintf(&ret, "%s *%s", c_out_types[arg->type], arg->name);
                break;
            case file_dt:
                printed_chars = asprintf(&ret, "%s %s", c_out_types[arg->type], arg->name);
                break;
            case array_char_dt:
            case array_byte_dt:
            case array_short_dt:
            case array_int_dt:
            case array_long_dt:
            case array_float_dt:
            case array_double_dt:
                printed_chars = asprintf(&ret, "%s* %s", c_out_types[arg->type], arg->name);
                break;
            case void_dt:
            case any_dt:
            case null_dt:
            default:
                ;
        }
    }

    if (printed_chars < 0) {
        printf("ERROR: Not possible to generate type and name string for argument.\n");
        free(ret);
        exit(1);
    }

    return ret;

}

/*
 * Return the string "return type" of the function
 */
static char* construct_returntype(function* func) {

    char* ret;
    int printed_chars = -1;

    if (func->return_type == int_dt) {
        printf("\t\t WARNING: Return type int_dt is not implemented, but permited.\n");
        printed_chars = asprintf(&ret, "int");
    }
    else if (func->return_type != void_dt) {
        if(func->return_type == object_dt || 
          (func->return_type >= array_char_dt && func->return_type <= array_double_dt)) {
            printed_chars = asprintf(&ret, "%s*", func->return_typename);
        } 
        else {
            printed_chars = asprintf(&ret, "%s", func->return_typename);
        }
    } 
    else {
        printed_chars = asprintf(&ret, "%s", c_types[func->return_type]);
    }

    if (printed_chars < 0) {
        printf("WARNING: Not possible to generate return type for function.");
        free(ret);

        return NULL;
    }

    return ret;

}

/*
 * Return the string "return type function name" of the function
 */
static char* construct_returntype_and_functionname(function* func) {

    char* ret;
    int printed_chars = -1;

    printed_chars = asprintf(&ret, "%s %s", construct_returntype(func), func->name);

    if (printed_chars < 0) {
        printf("ERROR: Not possible to generate return type and function name for function.");
        free(ret);
        exit(1);
    }

    return ret;

}

/*
 * Generate task prototype, composed of return type, function name and all the
 * arguments of the function implementing the task.
 */
static void generate_prototype(FILE *outFile, function *current_function) {

    argument *current_argument = current_function->first_argument;

    fprintf(outFile, "%s(", construct_returntype_and_functionname(current_function));
    while (current_argument != NULL) {
        fprintf(outFile, "%s", construct_type_and_name(current_argument));
        current_argument = current_argument->next_argument;
        if (current_argument != NULL) {
            fprintf(outFile, ", ");
        }
    }
    fprintf(outFile, ")");
}

//TODO: Maybe not needed

/*static void serialize_array(FILE *outFile, argument *arg){

    char* type;
    switch (arg->type) {
        case array_char_dt:
            type="char";
            break;
        case array_byte_dt:
            type="unsigned char";
            break;
        case array_short_dt:
            type="short";
            break;
        case array_int_dt:
            type="int";
            break;
        case array_long_dt:
            type="long";
            break;
        case array_float_dt:
            type="float";
            break;
        case array_double_dt:
            type="double";
            break;
        default:
            type=arg->classname;
    }
    fprintf(outFile, "\t\t ofstream %s_ofs(%s_filename, std::ofstream::trunc | std::ios::binary);\n", arg->name, arg->name);
    fprintf(outFile, "\t\t %s_ofs.write ((char*)%s, sizeof(%s)*%s);\n", arg->name, arg->name, type, arg->elements);
    //fprintf(outFile, "\t\t %s_ofs.flush();\n", arg->name);
    fprintf(outFile, "\t\t %s_ofs.close();\n", arg->name);
}

static void deserialize_array(FILE *outFile, argument *arg){

    char* type;
    switch (arg->type) {
        case array_char_dt:
            type="char";
            break;
        case array_byte_dt:
            type="unsigned char";
            break;
        case array_short_dt:
            type="short";
            break;
        case array_int_dt:
            type="int";
            break;
        case array_long_dt:
            type="long";
            break;
        case array_float_dt:
            type="float";
            break;
        case array_double_dt:
            type="double";
            break;
        default:
            type=arg->classname;
    }
    fprintf(outFile, "\t\t ifstream %s_ifs(%s_filename, std::ios::binary);\n", arg->name, arg->name);
    fprintf(outFile, "\t\t %s_ifs.read((char*)%s, sizeof(%s)*%s);\n", arg->name, arg->name, type, arg->elements);
    fprintf(outFile, "\t\t %s_ifs.close();\n", arg->name);
}
*/

static void add_serialization(FILE *outFile, enum datatype type, char* object, char* filename, char *elements, char *tabs) {
    if (type == string_dt || type == wstring_dt || type== string_64_dt) {
        fprintf(outFile, "%s  int %s_res = compss_object_serialize(%s, %s);\n",tabs, object, object, filename);
    } else if(type >= array_char_dt && type <= array_double_dt) {
        fprintf(outFile, "%s  int %s_res = compss_array_serialize(%s, %s, %s);\n",tabs, object, object, filename, elements);
    } else {
        fprintf(outFile, "%s  int %s_res = compss_object_serialize(%s, %s);\n",tabs, object, object, filename);
    }
    fprintf(outFile, "%s if (%s_res != 0){\n", tabs, object);
    fprintf(outFile, "%s\t cout << \"[C Binding] Error serializing from memory.\" << endl;\n", tabs);
    fprintf(outFile, "%s\t return %s_res;\n", tabs, object);
    fprintf(outFile, "%s }\n", tabs);
}

static void add_deserialization(FILE *outFile, enum datatype type, char* object, char* filename, char *elements, char *tabs) {
    if (type == string_dt || type == wstring_dt || type== string_64_dt) {
        fprintf(outFile, "%s  int %s_res = compss_object_deserialize(%s, %s);\n",tabs, object, object, filename);
    } else if(type >= array_char_dt && type <= array_double_dt) {
        fprintf(outFile, "%s  int %s_res = compss_array_deserialize(%s, %s, %s);\n",tabs, object, object, filename, elements);
    } else {
        fprintf(outFile, "%s  int %s_res = compss_object_deserialize(%s, %s);\n",tabs, object, object, filename);
    }
    fprintf(outFile, "%s if (%s_res != 0){\n", tabs, object);
    fprintf(outFile, "%s\t cout << \"[C Binding] Error deserializing from memory.\" << endl;\n", tabs);
    fprintf(outFile, "%s\t return %s_res;\n", tabs, object);
    fprintf(outFile, "%s } \n", tabs);
}

static void add_copy(FILE *outFile, enum datatype type, char* object_from, char* object_to, char *elements, char *tabs) {
    if (type == string_dt || type == wstring_dt || type== string_64_dt) {
        fprintf(outFile, "%s { int %s_res = compss_object_copy(%s, %s);\n", tabs, object_from, object_from, object_to);
    } else if(type >= array_char_dt && type <= array_double_dt) {
        fprintf(outFile, "%s { int %s_res = compss_array_copy(%s, %s, %s);\n",tabs, object_from, object_from, object_to, elements);
    } else {
        fprintf(outFile, "%s { int %s_res = compss_object_copy(%s, %s);\n",tabs, object_from, object_from, object_to);
    }
    fprintf(outFile, "%s if (%s_res != 0){\n", tabs, object_from);
    fprintf(outFile, "%s\t cout << \"[C Binding] Error deserializing from memory.\" << endl;\n", tabs);
    fprintf(outFile, "%s\t return %s_res;\n", tabs, object_from);
    fprintf(outFile, "%s } }\n", tabs);
}

/*Currently not used
static void add_arg_serialization(FILE *outFile, argument *arg, char* object_suffix, char* file_suffix, char *tabs){
    char * objectname = concat(arg->name, object_suffix);
    char * filename = concat(arg->name, file_suffix);
    add_serialization(outFile, arg->type, objectname, filename, arg->elements, tabs);
    free(objectname);
    free(filename);
}

static void add_arg_deserialization(FILE *outFile, argument *arg, char* object_suffix, char* file_suffix, char* tabs){
    char * objectname = concat(arg->name, object_suffix);
    char * filename = concat(arg->name, file_suffix);
    add_deserialization(outFile, arg->type, objectname, filename, arg->elements, tabs);
    free(objectname);
    free(filename);
}*/

static void generate_execute_empty(FILE *outFile) {
    generate_executor_prototype(outFile);
    fprintf(outFile, "return 0;\n");
    fprintf(outFile, "}\n");
}

static void generate_remove_func(FILE *outFile, Types current_types) {
    int i;
    fprintf(outFile, "int CBindingCache::removeData(compss_pointer cp){\n");
    fprintf(outFile, "\t switch(cp.type){\n");
    for (i=0; i< current_types.num; i++) {
        // Remove treatment
        char * dataType = current_types.types[i].name;
        fprintf(outFile, "\t\t case %d:\n", i);
        fprintf(outFile, "\t\t\t if (cp.pointer != NULL) {\n ");
        //fprintf(outFile, "\t\t\t\t %s *data_%d = (%s*)cp.pointer;\n", dataType, i, dataType);
        fprintf(outFile, "\t\t\t\t if(is_debug()){\n"); //Open if debug
        fprintf(outFile, "\t\t\t\t\t cout << \"[C Binding] Deleting object from memory \" << hex << cp.pointer << endl;\n");
        fprintf(outFile, "\t\t\t\t }\n");// end if debug;
        /*fprintf(outFile, "\t\t\t\t if (data_%d != NULL){\n",i);
        fprintf(outFile, "\t\t\t\t\t delete(data_%d);\n",i);
        fprintf(outFile, "\t\t\t\t }\n");*/
        fprintf(outFile, "\t\t\t\t if (cp.elements > 0 ){\n");
        fprintf(outFile, "\t\t\t\t\t cout << \"[C Binding] Deleting array\" << endl;\n");
        fprintf(outFile, "\t\t\t\t\t free(cp.pointer);\n");
        fprintf(outFile, "\t\t\t\t } else {\n");
        fprintf(outFile, "\t\t\t\t\t cout << \"[C Binding] Deleting object\" << endl;\n");
        fprintf(outFile, "\t\t\t\t\t delete((%s*) cp.pointer);\n", dataType);
        fprintf(outFile, "\t\t\t\t }\n");// end if cp.elements
        fprintf(outFile, "\t\t\t\t if(is_debug()){\n"); //Open if debug
        fprintf(outFile, "\t\t\t\t\t cout << \"[C Binding] Object has been removed.\" << endl;\n");
        fprintf(outFile, "\t\t\t\t }\n");// end if debug;
        fprintf(outFile, "\t\t\t }\n");
        fprintf(outFile, "\t\t\t break;\n");
    }
    fprintf(outFile, "\t\t default:;\n");
    fprintf(outFile, "\t }\n");
    fprintf(outFile, "\t return 0;");
    fprintf(outFile, "}\n");
    fprintf(outFile, "\n");
}

static void generate_copy_func(FILE *outFile, Types current_types) {
    int i;
    fprintf(outFile, "int CBindingCache::copyData(compss_pointer from, compss_pointer &to){\n");
    fprintf(outFile, "\t switch(from.type){\n");
    for (i=0; i< current_types.num; i++) {
        //remove treatment
        char * dataType = current_types.types[i].name;
        char *object_from = malloc(25);
        snprintf(object_from, 25, "from_%d", i);
        char *object_to = malloc(25);
        snprintf(object_to, 25, "to_%d", i);
        fprintf(outFile, "\t\t case %d:\n", i);
        fprintf(outFile, "\t\t\t %s *%s;\n", dataType, object_from);
        fprintf(outFile, "\t\t\t %s *%s;\n", dataType, object_to);
        fprintf(outFile, "\t\t\t %s = (%s*)from.pointer;\n", object_from, dataType);
        add_copy(outFile,current_types.types[i].dt, object_from, object_to, "from.elements", "\t\t\t");
        fprintf(outFile, "\t\t\t to.pointer = (void*)%s;\n", object_to);
        fprintf(outFile, "\t\t\t to.elements = from.elements;\n");
        fprintf(outFile, "\t\t\t to.type = from.type;\n");
        fprintf(outFile, "\t\t\t to.size = from.size;\n");
        fprintf(outFile, "\t\t\t break;\n");
    }
    fprintf(outFile, "\t\t default:;\n");
    fprintf(outFile, "\t }\n");
    fprintf(outFile, "\t return 0;");
    fprintf(outFile, "}\n");
    fprintf(outFile, "\n");
}

static void generate_serializeToStream_func(FILE *outFile, Types current_types) {
    int i;
    fprintf(outFile, "int CBindingCache::serializeToStream(compss_pointer cp, JavaNioConnStreamBuffer &jsb){\n");
    fprintf(outFile, "\t switch(cp.type){\n");

    for (i=0; i< current_types.num; i++) {
        //serialize treatment
        Type type = current_types.types[i];
        char *object_name = malloc(25);
        snprintf(object_name, 25, "data_%d", i);
        char * dataType = type.name;
        fprintf(outFile, "\t\t case %d:\n", i);
        fprintf(outFile, "\t\t {");
        fprintf(outFile, "\t\t\t if(is_debug()){\n"); //Open if debug
        fprintf(outFile, "\t\t\t\t cout << \"[C Binding] Object %s is about to be serialized.\" << endl << flush;\n", object_name);
        fprintf(outFile, "\t\t\t }");// end if debug;
        fprintf(outFile, "\t\t\t %s *%s;\n",dataType, object_name);
        fprintf(outFile, "\t\t\t %s = (%s*)cp.pointer;\n", object_name, dataType);
        add_serialization(outFile, type.dt, object_name, "jsb","cp.elements", "\t\t\t\t");
        fprintf(outFile, "\t\t\t if(is_debug()){\n"); //Open if debug
        fprintf(outFile, "\t\t\t\t cout << \"[C Binding] Object %s has been serialized.\" << endl << flush;\n", object_name);
        fprintf(outFile, "\t\t\t }");// end if debug;
        fprintf(outFile, "\t\t\t break;\n");
        fprintf(outFile, "\t\t }");
        free(object_name);
    }

    fprintf(outFile, "\t }\n");
    fprintf(outFile, "\t return 0;\n");
    fprintf(outFile, "}\n");
    fprintf(outFile, "\n");
}

static void generate_serializeToFile_func(FILE *outFile, Types current_types) {
    int i;
    fprintf(outFile, "int CBindingCache::serializeToFile(compss_pointer cp, const char * filename){\n");
    fprintf(outFile, "\t switch(cp.type){\n");

    for (i=0; i< current_types.num; i++) {
        //serialize treatment
        Type type = current_types.types[i];
        char *object_name = malloc(25);
        snprintf(object_name, 25, "data_%d", i);
        char * dataType = type.name;
        fprintf(outFile, "\t\t case %d:\n", i);
        fprintf(outFile, "\t\t {");
        fprintf(outFile, "\t\t\t if(is_debug()){\n"); //Open if debug
        fprintf(outFile, "\t\t\t\t cout << \"[C Binding] Object %s is about to be serialized.\" << endl << flush;\n", object_name);
        fprintf(outFile, "\t\t\t }");// end if debug;
        fprintf(outFile, "\t\t\t %s *%s;\n",dataType, object_name);
        fprintf(outFile, "\t\t\t %s = (%s*)cp.pointer;\n", object_name, dataType);
        add_serialization(outFile, type.dt, object_name, "filename", "cp.elements", "\t\t\t\t");
        fprintf(outFile, "\t\t\t if(is_debug()){\n"); //Open if debug
        fprintf(outFile, "\t\t\t\t cout << \"[C Binding] Object %s has been serialized.\" << endl << flush;\n", object_name);
        fprintf(outFile, "\t\t\t }");// end if debug;
        fprintf(outFile, "\t\t\t break;\n");
        fprintf(outFile, "\t\t }");
        free(object_name);
    }

    fprintf(outFile, "\t }\n");
    fprintf(outFile, "\t return 0;\n");
    fprintf(outFile, "}\n");
    fprintf(outFile, "\n");
}

static void generate_deserializeFromStream_func(FILE *outFile, Types current_types) {
    int i;
    fprintf(outFile, "int CBindingCache::deserializeFromStream(JavaNioConnStreamBuffer &jsb, compss_pointer &cp){\n");
    fprintf(outFile, "\t switch(cp.type){\n");

    for (i=0; i< current_types.num; i++) {
        //serialize treatment
        Type type = current_types.types[i];
        char *object_name = malloc(25);
        snprintf(object_name, 25, "data_%d", i);
        char * dataType = type.name;
        fprintf(outFile, "\t\t case %d:\n", i);
        fprintf(outFile, "\t\t {");
        fprintf(outFile, "\t\t\t if(is_debug()){\n"); //Open if debug
        fprintf(outFile, "\t\t\t\t cout << \"[C Binding] Object %s is about to be deserialized.\" << endl << flush;\n", object_name);
        fprintf(outFile, "\t\t\t }");// end if debug;
        if (type.dt == object_dt) {
            fprintf(outFile, "\t\t\t %s *%s = new %s();\n",dataType, object_name, dataType);
        } else {
            fprintf(outFile, "\t\t\t %s *%s = new %s[cp.elements];\n",dataType, object_name, dataType);
        }
        fprintf(outFile, "\t\t\t cp.pointer = %s;\n", object_name);
        add_deserialization(outFile,type.dt, object_name, "jsb", "cp.elements", "\t\t\t\t");
        fprintf(outFile, "\t\t\t if(is_debug()){\n"); //Open if debug
        fprintf(outFile, "\t\t\t\t cout << \"[C Binding] Object %s has been deserialized.\" << endl << flush;\n", object_name);
        fprintf(outFile, "\t\t\t }");// end if debug;
        fprintf(outFile, "\t\t\t break;\n");
        fprintf(outFile, "\t\t }");
        free(object_name);
    }

    fprintf(outFile, "\t }\n");
    fprintf(outFile, "\t return 0;\n");
    fprintf(outFile, "}\n");
    fprintf(outFile, "\n");
}

static void generate_deserializeFromFile_func(FILE *outFile, Types current_types) {
    int i;
    fprintf(outFile, "int CBindingCache::deserializeFromFile(const char * filename, compss_pointer &cp){\n");
    fprintf(outFile, "\t switch(cp.type){\n");

    for (i=0; i< current_types.num; i++) {
        //serialize treatment
        Type type = current_types.types[i];
        char *object_name = malloc(25);
        snprintf(object_name, 25, "data_%d", i);
        char * dataType = type.name;
        fprintf(outFile, "\t\t case %d:\n", i);
        fprintf(outFile, "\t\t {");
        fprintf(outFile, "\t\t\t if(is_debug()){\n"); //Open if debug
        fprintf(outFile, "\t\t\t\t cout << \"[C Binding] Object %s is about to be deserialized.\" << endl << flush;\n", object_name);
        fprintf(outFile, "\t\t\t }");// end if debug;
        if (type.dt == object_dt) {
            fprintf(outFile, "\t\t\t %s *%s = new %s();\n",dataType, object_name, dataType);
        } else {
            fprintf(outFile, "\t\t\t %s *%s = new %s[cp.elements];\n",dataType, object_name, dataType);
        }
        //fprintf(outFile, "\t\t\t cp.pointer = %s;\n", object_name);
        add_deserialization(outFile,type.dt, object_name, "filename","cp.elements", "\t\t\t\t");
        fprintf(outFile, "\t\t\t cp.pointer = %s;\n", object_name);
        fprintf(outFile, "\t\t\t if(is_debug()){\n"); //Open if debug
        fprintf(outFile, "\t\t\t\t cout << \"[C Binding] Object %s has been deserialized.\" << endl << flush;\n", object_name);
        fprintf(outFile, "\t\t\t }");// end if debug;
        fprintf(outFile, "\t\t\t break;\n");
        fprintf(outFile, "\t\t }");
        free(object_name);
    }

    fprintf(outFile, "\t }\n");
    fprintf(outFile, "\t return 0;\n");
    fprintf(outFile, "}\n");
    fprintf(outFile, "\n");
}

static void include_header(include* current_include) {
    fprintf(includeFile, "#include <%s>\n", current_include->name);
}

static void add_include_headers(include* first_include) {
    include* include = first_include;

    while (include != NULL) {
        include_header(first_include);
        include = include->next_include;
    }
}

/*
 * Generate includes of task classes in header files
 */
static void generate_class_includes_and_check_types(FILE *outFile, Types *current_types, function *current_function) {

    argument *current_argument;
    if (current_function->classname != NULL) {
        Type t;
        t.name = current_function->classname;
        t.dt = object_dt;
        t.elements = "0";
        if (!containsType(t, *(current_types))) {
            addType(t, current_types);
            fprintf(outFile, "#include \"%s.h\"\n", current_function->classname);
        }
    }
    if (current_function->return_type == object_dt) {
        Type t;
        t.name = current_function->return_typename;
        t.dt = current_function->return_type;
        t.elements = "0";
        if (!containsType(t, *(current_types))) {
            addType(t, current_types);
            fprintf(outFile, "#include \"%s.h\"\n", current_function->return_typename);
        }
    } else if(current_function->return_type >= array_char_dt && current_function->return_type <= array_double_dt) {
        Type t;
        t.name = current_function->return_typename;
        t.elements = current_function->return_elements;
        t.dt = current_function->return_type;
        if (!containsType(t, *(current_types))) {
            addType(t, current_types);
        }
    }
    current_argument = current_function->first_argument;
    while (current_argument != NULL) {
        if (current_argument->type == object_dt) {
            Type t;
            t.name = current_argument->classname;
            t.dt = current_argument->type;
            t.elements ="0";
            if (!containsType(t, *(current_types))) {
                addType(t, current_types);
                fprintf(outFile, "#include \"%s.h\"\n", current_argument->classname);
            }
        } else if(current_argument->type >= array_char_dt && current_argument->type <= array_double_dt) {
            Type t;
            t.name = current_argument->classname;
            t.elements = current_argument->elements;
            t.dt = current_argument->type;
            if (!containsType(t, *(current_types))) {
                addType(t, current_types);
            }
        }
        current_argument = current_argument->next_argument;
    }
    printf("Detected types after function %s:\n",current_function->name);
    printAllTypes(*(current_types));
}

/*
 * Generate buffer to add task parameters before calling the task execution
 */
static void generate_parameter_buffers(FILE *outFile, function *func) {
	fprintf(outFile, "\t debug_printf(\"Detected task %s\\n\");\n", func->name);
	int k = 0;
    //There is a target object
    if (( func->classname != NULL ) && (func->access_static == 0)) k = k + BUFF_ELEMENTS;
    //There is a return type
    if ( func->return_type != void_dt ) k = k + BUFF_ELEMENTS;

    fprintf(outFile, "\t void *arrayObjs[%d];\n", k + func->argument_count * BUFF_ELEMENTS);
    fprintf(outFile, "\t int found;\n");
    fprintf(outFile, "\n");
}

static void add_parameter_to_taskbuffer(FILE *outFile, char* name, enum datatype dt, enum direction dir, int param3, char* param4, int i, char* tabs) {
    fprintf(outFile, "%s arrayObjs[%d] = &%s;\n",tabs, i, name);
    fprintf(outFile, "%s int param%d = %d;\n", tabs, i+1, dt);
    fprintf(outFile, "%s arrayObjs[%d] = &param%d;\n", tabs, i+1, i+1);
    fprintf(outFile, "%s int param%d = %d;\n", tabs, i+2, dir);
    fprintf(outFile, "%s arrayObjs[%d] = &param%d;\n", tabs, i+2, i+2);
    fprintf(outFile, "%s int param%d = %d;\n", tabs, i+3, param3);
    fprintf(outFile, "%s arrayObjs[%d] = &param%d;\n", tabs, i+3,  i+3);
    fprintf(outFile, "%s char *param%d = \"null\";\n", tabs, i+4);
    fprintf(outFile, "%s arrayObjs[%d] = &param%d;\n", tabs, i+4, i+4);
    fprintf(outFile, "%s char *param%d = \"null\";\n", tabs, i+5);
    fprintf(outFile, "%s arrayObjs[%d] = &param%d;\n", tabs, i+5, i+5);
    fprintf(outFile, "%s char *param%d = \"null\";\n", tabs, i+6);
    fprintf(outFile, "%s arrayObjs[%d] = &param%d;\n", tabs, i+6, i+6);
    fprintf(outFile, "%s char *param%d = \"1.0\";\n", tabs, i+7);
    fprintf(outFile, "%s arrayObjs[%d] = &param%d;\n", tabs, i+7, i+7);
    fprintf(outFile, "%s int param%d = 0;\n", tabs, i+8);
    fprintf(outFile, "%s arrayObjs[%d] = &param%d;\n", tabs, i+8, i+8);
}

static void add_object_or_array_arg_master_treatment(FILE *outFile, argument *arg, int i, Types current_types) {
    Type t;
    t.name = arg->classname;
    t.elements = arg->elements;
    t.dt = arg->type;
    int position = getTypeNumber(t, current_types);
    if (position < 0) {
        printf("ERROR: Position for type (%s,%s,%d) not found", arg->classname, arg->elements, arg->type);
        printAllTypes(current_types);
        exit(1);
    }
    fprintf(outFile, "\t if (%s==NULL){\n", arg->name);
    if ( arg->dir == out_dir) {
        if (arg->type == object_dt) {
            fprintf(outFile, "\t\t %s = new %s();\n",arg->name, arg->classname);
        } else if (arg->type >= array_char_dt && arg->type <= array_double_dt) {
            fprintf(outFile, "\t\t %s = new %s[%s];\n",arg->name, arg->classname, arg->elements);
        }
    } else {
        fprintf(outFile, "\t\t printf(\"ERROR: %s is null.\");\n\t\t fflush(NULL);\n\t\t GS_Off(1);\n\t\t exit(1);", arg->name);
    }
    fprintf(outFile, "\t }\n");
    fprintf(outFile, "\t // Add object treatment for argument %s;\n", arg->name);
    fprintf(outFile, "\t char *%s_filename;\n", arg->name);
    fprintf(outFile, "\t compss_pointer %s_cp;\n",arg->name);
    fprintf(outFile, "\t %s_cp.type = %d;\n",arg->name, position);
    fprintf(outFile, "\t %s_cp.elements = %s;\n",arg->name, arg->elements);
    //TODO: Size currently managed in arrays. Add object size when serializing
    fprintf(outFile, "\t %s_cp.size = %s_cp.elements * sizeof(%s);\n",arg->name, arg->name, arg->classname);
    fprintf(outFile, "\t %s_cp.pointer = (void*)%s;\n",arg->name, arg->name);
    //fprintf(outFile, "\t if (is_persistent()){\n");
    fprintf(outFile, "\t\t found = compss_register(&(*%s), (datatype)%d, (direction)%d, \"%s\", %s_filename, %s_cp.type, %s_cp.elements);\n", arg->name, arg->type, arg->dir, arg->classname, arg->name, arg->name, arg->name);
    /*fprintf(outFile, "\t } else {\n");
    fprintf(outFile, "\t\t found = compss_register(&(*%s), (datatype)%d, (direction)%d, \"%s\", %s_filename, %s_cp.type, %s_cp.elements);\n", arg->name, file_dt, arg->dir, arg->classname, arg->name, arg->name, arg->name);
    fprintf(outFile, "\t } \n");*/
    if (( arg->dir == in_dir) || (arg->dir == inout_dir)) {
        fprintf(outFile, "\t if (!found) {\n");
        //Changed for M2M
        //fprintf(outFile, "\t\t if (is_persistent()){\n");
        fprintf(outFile, "\t\t\t debug_printf(\"[Generated C-Binding] Object not found in registry. \\n\\tStoring %%s to cache with pointer %%p type %%d elements %%d and size %%ld\\n\", %s_filename,(void*)%s, %s_cp.type, %s_cp.elements, %s_cp.size );\n", arg->name, arg->name, arg->name, arg->name, arg->name);
        fprintf(outFile, "\t\t\t get_cache()->storeInCache(%s_filename,%s_cp);\n", arg->name, arg->name);
        /*fprintf(outFile, "\t\t } else {\n");
        fprintf(outFile, "\t\t debug_printf(\"Object not found in registry. Serializing to %%s \\n\", %s_filename);\n" , arg->name);
        add_arg_serialization(outFile, arg, "", "_filename", "\t\t\t");
        fprintf(outFile, "\t\t } \n");*/
        fprintf(outFile, "\t }\n");
    }
    fprintf(outFile, "\t ostringstream %s_oss;\n", arg->name);
    fprintf(outFile, "\t %s_oss << %s_filename << \":%d:\" << %s;\n", arg->name, arg->name, position, arg->elements);
    fprintf(outFile, "\t char *%s_value=strdup(%s_oss.str().c_str());\n", arg->name, arg->name);
    fprintf(outFile, "\t debug_printf(\"Binding Value: %%s (before %%s)\\n\", %s_value,%s_oss.str().c_str());\n",  arg->name, arg->name);
    char* name = concat(arg->name,"_value");
    //fprintf(outFile, "\t if (is_persistent()){\n");
    add_parameter_to_taskbuffer(outFile, name, binding_object_dt, arg->dir, 3, "\"null\"", i, "\t\t");
    /*fprintf(outFile, "\t } else {\n");
    name = concat(arg->name,"_filename");
    add_parameter_to_taskbuffer(outFile, name, file_dt, arg->dir, 3, "\"null\"", i, "\t");
    fprintf(outFile, "\t }\n");*/

}

static void treat_master_argument(FILE *outFile, argument *arg, int i, Types current_types) {
    enum datatype t = arg->type; //Used only in the enum_dt, wstring_dt case...
    switch (t) {
    case enum_dt:
        t = int_dt;
    case char_dt:
    case wchar_dt:
    case boolean_dt:
    case short_dt:
    case long_dt:
    case longlong_dt:
    case int_dt:
    case float_dt:
    case double_dt:
    case string_dt:
    case string_64_dt:
    case wstring_dt:
        if ( arg->dir == in_dir) {
            fprintf(outFile, "\t // Add treatment for argument %s;\n", arg->name);
            add_parameter_to_taskbuffer(outFile, arg->name, t, arg->dir, 3, "\"null\"", i, "\t");
        } else {
            add_object_or_array_arg_master_treatment(outFile, arg, i, current_types);
        }
        break;
    case file_dt:
        fprintf(outFile, "\t // Add file treatment for argument %s;\n", arg->name);
        add_parameter_to_taskbuffer(outFile, arg->name, arg->type, arg->dir, 3, "\"null\"", i, "\t");
        break;
    case object_dt:
        add_object_or_array_arg_master_treatment(outFile,arg,i, current_types);
        break;
    case array_char_dt:
    case array_byte_dt:
    case array_short_dt:
    case array_int_dt:
    case array_long_dt:
    case array_float_dt:
    case array_double_dt:
        add_object_or_array_arg_master_treatment(outFile,arg,i, current_types);
        break;
    case void_dt:
    case any_dt:
    case null_dt:
    default:
        ;
    }
}

static void generate_parameter_marshalling(FILE *outFile, function *func, Types current_types) {
    argument *arg;
    int j = 0;
    int i = 0;

    arg = func->first_argument;
    while (arg != NULL) {
        i = j*BUFF_ELEMENTS;

        fprintf(outFile, "\n");

        treat_master_argument(outFile, arg, i, current_types);

        arg = arg->next_argument;
        j++;
    }

    fprintf(outFile, "\n");

    if ((func->classname != NULL) && (func->access_static == 0)) {
        i = j*BUFF_ELEMENTS;

        argument th;
        th.name="this";
        th.type=object_dt;
        th.dir=inout_dir;
        th.classname=func->classname;
        th.elements="0";

        add_object_or_array_arg_master_treatment(outFile, &th, i, current_types);

        fprintf(outFile, "\n");

        j++;
    }

    fprintf(outFile, "\n");

    if (func->return_type == int_dt) {
        printf("\t\t WARNING: Return type int_dt is not implemented, but permited.\n");
    }
    else if (func->return_type != void_dt) {
        i = j*BUFF_ELEMENTS;

        argument ret;
        ret.name="return_obj";
        ret.type= func->return_type;
        ret.dir=out_dir;
        ret.classname=func->return_typename;

        if (func->return_elements != NULL) {
            ret.elements=func->return_elements;
        } else {
            ret.elements="0";
        }

        fprintf(outFile, "\t %s *%s = NULL;\n", ret.classname, ret.name);

        treat_master_argument(outFile, &ret, i, current_types);

        fprintf(outFile, "\n");

        j++;
    }

}

/*
 * Generate master task execution
 */
static void generate_execute_task_call(FILE *outFile, function *func) {
    char *class_name = strdup("NULL");
    char *on_failure = strdup("RETRY");
    int time_out = 0;
    int priority = 0; //default false
    int num_nodes = 1;
    int is_reduce = 0; // default false
    int reduce_chunk = 0;
    int is_replicated = 0; // default false
    int is_distributed = 0; // default false 
    int has_target = 0; //default false
    int arg_count = func->argument_count;
    if ( func->classname != NULL ) {
        class_name = func->classname;
        if (func->access_static == 0) {
            arg_count++;
            has_target = 1; //set true
        }
    }
    int num_returns = 0;
    if (func->return_type == int_dt) {
        printf("\t\t WARNING: Return type int_dt is not implemented, but permited.\n");
    }
    else if (func->return_type != void_dt) {
        arg_count++;
    	num_returns = 1;
    }

    fprintf(outFile, "\t char *method_name = strdup(\"%s\");\n", func->name);
    //ExecuteTask params: appId, className, onFailure, timeout,  method_name, priority, num_nodes, is_reduce, reduce_chunk, is_replicated, is_distributed, has_target,  num_returns, num_params params
    fprintf(outFile, "\t GS_ExecuteTask(0L, \"%s\", \"%s\", %d, method_name, %d, %d, %d, %d, %d, %d, %d, %d, %d, (void**)arrayObjs);\n", 
		    class_name, on_failure, time_out, priority, num_nodes, is_reduce, reduce_chunk, is_replicated, is_distributed, has_target, num_returns, arg_count);
    fprintf(outFile, "\t debug_printf(\"[   BINDING]  -  @%%s  -  Task submited in the runtime\\n\", method_name);\n");
    fprintf(outFile, "\n");
    
    fprintf(outFile, "\t debug_printf(\"[   BINDING]  -  @%%s  -  Free method name\\n\", method_name);\n");
    fprintf(outFile, "\t free(method_name);\n");
    fprintf(outFile, "\t debug_printf(\"[   BINDING]  -  End of task submission.\\n\");\n");

    if (func->return_type == int_dt) {
        printf("\t\t WARNING: Return type int_dt is not implemented, but permited.\n");
    }
    else if (func->return_type != void_dt) {
        fprintf(outFile, "\n\t return return_obj;\n");
        fprintf(outFile, "\n");
    }

}

static void add_checkinCache_and_management(FILE *outFile, const char* name, const char* dataType, const char * elements, int position, bool preserveData) {
    //Check if dest is in cache
    fprintf(outFile, "\t\t\t\t if(is_debug()) cout << \"[C Binding] Checking if object %s with id \"<< %s_dest_id_str << \" is in cache.\" << endl << flush;\n", name, name);
    fprintf(outFile, "\t\t\t\t bool found = cache->isInCache(%s_dest_id);\n", name);
    //If dest id not found -> try if origin is in cache
    fprintf(outFile, "\t\t\t\t if (!found){\n");
    fprintf(outFile, "\t\t\t\t\t if(is_debug()) cout << \"[C Binding] Destination id not found in cache. Checking if object %s with id \"<< %s_orig_id_str << \" is in cache.\" << endl << flush;\n", name, name);
    fprintf(outFile, "\t\t\t\t\t found = cache->isInCache(%s_orig_id);\n", name);

    //If origin id not found -> deserialize form file
    fprintf(outFile, "\t\t\t\t\t if (!found){\n");
    fprintf(outFile, "\t\t\t\t\t\t cout << \"[C Binding] Deserializing object %s from \" << %s_dest_id_str << \" as \" << %s_dest_id_str << endl << flush;\n", name, name, name);
    //New cache
    fprintf(outFile, "\t\t\t\t\t\t compss_pointer cp;\n");
    fprintf(outFile, "\t\t\t\t\t\t cp.type= %d;\n", position);
    fprintf(outFile, "\t\t\t\t\t\t cp.elements= %s_elements;\n", name);
    fprintf(outFile, "\t\t\t\t\t\t cp.size = cp.elements * sizeof(%s);\n", dataType);
    fprintf(outFile, "\t\t\t\t\t\t int res = cache->pullFromFile(%s_dest_id, %s_dest_id, cp);\n", name, name);

    fprintf(outFile, "\t\t\t\t\t\t if (res != 0){\n"); // If error pulling from dest_id -> try with origin id
    fprintf(outFile, "\t\t\t\t\t\t\t if(is_debug()) cout << \"[C Binding] Error Deserializing object %s as \"<< %s_dest_id_str <<\". Deserializing object %s from \" << %s_orig_id_str << \" as \" << %s_dest_id_str << endl << flush;\n", name, name, name, name, name);
    fprintf(outFile, "\t\t\t\t\t\t\t res = cache->pullFromFile(%s_dest_id, %s_orig_id, cp);\n", name, name);

    fprintf(outFile, "\t\t\t\t\t\t\t if (res != 0){\n"); // If error pulling from origin_id -> return error
    fprintf(outFile, "\t\t\t\t\t\t\t\t cerr << \"[C Binding] Error Deserializing object %s as \"<< %s_orig_id_str << endl << flush;\n", name, name);
    fprintf(outFile, "\t\t\t\t\t\t\t\t return res;\n");
    fprintf(outFile, "\t\t\t\t\t\t\t }\n");
    fprintf(outFile, "\t\t\t\t\t\t }\n");
    fprintf(outFile, "\t\t\t\t\t\t %s = (%s*)cp.pointer;\n", name, dataType);
    fprintf(outFile, "\t\t\t\t\t\t if(is_debug()) cout << \"[C Binding] Object %s has been added to the cache as \" << %s_dest_id_str << \" with type %d.\"  << endl << flush;\n", name, name,position);

    //if found with origin id -> move or copy in cache to dest id evaluating preserve data.
    fprintf(outFile, "\t\t\t\t\t } else {  // found as origin id -> move or copy \n");
    if (preserveData) {
        // if preserve data -> make a memory copy
        fprintf(outFile, "\t\t\t\t\t\t if (string(%s_pres_data) == \"true\"){\n", name);
        fprintf(outFile, "\t\t\t\t\t\t\t if(is_debug()) cout << \"[C Binding] Copying object %s in cache. From \" << %s_orig_id << \" to \"<< %s_dest_id << endl << flush;\n", name, name, name);
        fprintf(outFile, "\t\t\t\t\t\t\t compss_pointer %s_to;\n", name);
        fprintf(outFile, "\t\t\t\t\t\t\t int res = cache->copyInCache(%s_orig_id, %s_dest_id, %s_to);\n", name, name, name);

        fprintf(outFile, "\t\t\t\t\t\t\t if (res != 0){\n"); //If error copying -> return error
        fprintf(outFile, "\t\t\t\t\t\t\t\t cerr << \"[C Binding] Error copying object %s in \"<< %s_orig_id << endl << flush;\n", name, name);
        fprintf(outFile, "\t\t\t\t\t\t\t\t return res;\n");
        fprintf(outFile, "\t\t\t\t\t\t\t }\n"); //end error
        fprintf(outFile, "\t\t\t\t\t\t\t %s = (%s*)%s_to.pointer;\n", name, dataType, name);

        // else -> make a cache move
        fprintf(outFile, "\t\t\t\t\t\t } else { \n");
        fprintf(outFile, "\t\t\t\t\t\t\t if(is_debug()) cout << \"[C Binding] Moving object %s in cache. From \" << %s_orig_id << \" to \"<< %s_dest_id << endl << flush;\n", name, name, name);
        fprintf(outFile, "\t\t\t\t\t\t\t int res = cache->moveInCache(%s_orig_id, %s_dest_id);\n", name, name);
        fprintf(outFile, "\t\t\t\t\t\t\t if (res != 0){\n"); //If error moving -> return error
        fprintf(outFile, "\t\t\t\t\t\t\t\t cerr << \"[C Binding] Error copying object %s in \"<< %s_orig_id << endl << flush;\n", name, name);
        fprintf(outFile, "\t\t\t\t\t\t\t\t return res;\n");
        fprintf(outFile, "\t\t\t\t\t\t\t }\n"); //End error
        // and get the object pointer from cache
        fprintf(outFile, "\t\t\t\t\t\t\t if(is_debug()) cout << \"[C Binding] Object %s has been added to the cache as\" << %s_dest_id_str << \" with type %d. Previous version is kept as \" << %s_orig_id_str << endl << flush;\n", name, name, position, name);
        fprintf(outFile, "\t\t\t\t\t\t\t compss_pointer cp;\n");
        fprintf(outFile, "\t\t\t\t\t\t\t res = cache->getFromCache(%s_dest_id, cp);\n", name);
        fprintf(outFile, "\t\t\t\t\t\t\t if (res != 0){\n"); //If error getting -> return error
        fprintf(outFile, "\t\t\t\t\t\t\t\t cerr << \"[C Binding] Error getting from cache object %s as \"<< %s_dest_id_str << endl << flush;\n", name, name);
        fprintf(outFile, "\t\t\t\t\t\t\t\t return res;\n");
        fprintf(outFile, "\t\t\t\t\t\t\t }\n"); //end error
        fprintf(outFile, "\t\t\t\t\t\t\t %s = (%s*)cp.pointer;\n", name, dataType);
        fprintf(outFile, "\t\t\t\t\t\t }\n"); //end else
    }
    fprintf(outFile, "\t\t\t\t\t } //end not found as origin \n"); // end else if not found as origin
    // if found as dest_id -> get from cache
    fprintf(outFile, "\t\t\t\t } else { // found as dest id -> get from cache\n");
    fprintf(outFile, "\t\t\t\t\t compss_pointer cp;\n");
    fprintf(outFile, "\t\t\t\t\t int res = cache->getFromCache(%s_dest_id, cp);\n", name);
    fprintf(outFile, "\t\t\t\t\t if (res != 0){\n"); // If error getting from cache -> return error
    fprintf(outFile, "\t\t\t\t\t\t cerr << \"[C Binding] Error getting from cache object %s as \"<< %s_dest_id_str << endl << flush;\n", name, name);
    fprintf(outFile, "\t\t\t\t\t\t return res;\n");
    fprintf(outFile, "\t\t\t\t\t }\n"); //end error
    fprintf(outFile, "\t\t\t\t\t %s = (%s*)cp.pointer;\n", name, dataType);
    fprintf(outFile, "\t\t\t\t\t if(is_debug()) cout << \"[C Binding] Object %s has been read from the cache as\" << %s_dest_id_str << endl << flush;\n", name, name);
    fprintf(outFile, "\t\t\t\t } //end found as dest id \n"); //end else

    //TODO Check if it is needed
    if (preserveData) {
        fprintf(outFile, "\t\t\t\t if (string(%s_pres_data) == \"false\"){\n", name);
        fprintf(outFile, "\t\t\t\t\t cache->deleteFromCache(%s_orig_id, false);\n", name);
        fprintf(outFile, "\t\t\t\t\t if(is_debug()) cout << \"[C Binding] Object \" << %s_orig_id_str << \" has been removed from the cache.\" << endl << flush;\n", name);
        fprintf(outFile, "\t\t\t\t }\n");
    }
}

static void add_object_or_array_arg_worker_treatment(FILE *outFile, argument *arg, int initOffset, Types current_types, bool is_return) {
    //fprintf(outFile, "\t\t\t cout << \"[C Binding] Treating object %s ...\" << endl << flush;\n", arg->name);
    fprintf(outFile, "\t\t\t arg_offset += %d;\n", initOffset);

    fprintf(outFile, "\t\t\t char* %s_filename_og = strdup(argv[arg_offset]);\n", arg->name);
    fprintf(outFile, "\t\t\t arg_offset ++;\n");
    fprintf(outFile, "\t\t\t if(is_debug()) cout <<\"[C Binding] Received value : \"<< %s_filename_og << endl;\n",arg->name);
    fprintf(outFile, "\t\t\t int %s_type = atoi(argv[arg_offset]);\n", arg->name);
    fprintf(outFile, "\t\t\t arg_offset ++;\n");
    fprintf(outFile, "\t\t\t int %s_elements = atoi(argv[arg_offset]);\n", arg->name);

    if (arg->dir == out_dir && !is_return) {
        if (arg->type == object_dt) {
            fprintf(outFile, "\t\t\t\t %s = new %s();\n",arg->name, arg->classname);
        } else {
            fprintf(outFile, "\t\t\t\t %s = new %s[%s_elements];\n",arg->name, arg->classname, arg->name);
        }
    }

    fprintf(outFile, "\t\t\t char* %s_filename, *%s_orig_id, *%s_dest_id, *%s_pres_data, *%s_write_data;\n", arg->name, arg->name, arg->name, arg->name, arg->name);
    fprintf(outFile, "\t\t\t %s_filename = %s_filename_og ;\n", arg->name, arg->name);
    fprintf(outFile, "\t\t\t %s_orig_id = strsep(&%s_filename,\":\");\n", arg->name, arg->name);

    //Check if there is data id information in the value
    fprintf(outFile, "\t\t\t if (%s_orig_id != NULL && %s_filename != NULL){ // Obtain object from cache\n", arg->name, arg->name);
    fprintf(outFile, "\t\t\t\t if(is_debug()) cout << \"[C Binding] Managing object data id for %s ...\" << endl << flush;\n", arg->name);
    fprintf(outFile, "\t\t\t\t %s_dest_id = strsep(&%s_filename,\":\");\n", arg->name, arg->name);
    fprintf(outFile, "\t\t\t\t %s_pres_data = strsep(&%s_filename,\":\");\n", arg->name, arg->name);
    fprintf(outFile, "\t\t\t\t %s_write_data = strsep(&%s_filename,\":\");\n", arg->name, arg->name);
    fprintf(outFile, "\t\t\t\t string %s_orig_id_str(%s_orig_id);\n", arg->name, arg->name);
    fprintf(outFile, "\t\t\t\t string %s_dest_id_str(%s_dest_id);\n", arg->name, arg->name);
    Type t;
    t.name = arg->classname;
    t.elements = arg->elements;
    t.dt = arg->type;
    int position = getTypeNumber(t, current_types);
    if (position < 0) {
        printf("ERROR: Position for type (%s,%s,%d) not found", arg->classname, arg->elements, arg->type);
        printAllTypes(current_types);
        exit(1);
    }
    switch (arg->dir) {
    case in_dir:
        //when argument is in
        add_checkinCache_and_management(outFile, arg->name, arg->classname, t.elements, position, false);
        break;
    case inout_dir:
        //when argument is inout
        add_checkinCache_and_management(outFile, arg->name, arg->classname, t.elements, position, true);

        fprintf(outFile, "\t\t\t\t if (serializeOuts < 1){\n");
        fprintf(outFile, "\t\t\t\t\t remove(%s_dest_id);\n", arg->name);
        fprintf(outFile, "\t\t\t\t\t if(is_debug()) cout << \"[C Binding] Removed file \" << %s_dest_id << endl;\n", arg->name);
        fprintf(outFile, "\t\t\t\t }\n");

        break;
    case out_dir:
        //when argument is out
        if (!is_return) {
            fprintf(outFile, "\t\t\t\t compss_pointer %s_cp;\n", arg->name);
            fprintf(outFile, "\t\t\t\t %s_cp.type= %d;\n", arg->name, position);
            fprintf(outFile, "\t\t\t\t %s_cp.elements= %s_elements;\n", arg->name, arg->name);
            fprintf(outFile, "\t\t\t\t %s_cp.pointer = (void*)%s;\n", arg->name, arg->name);
            fprintf(outFile, "\t\t\t\t %s_cp.size = %s_cp.elements*sizeof(%s);\n", arg->name, arg->name, arg->classname);
            fprintf(outFile, "\t\t\t\t int res = cache->storeInCache(%s_dest_id, %s_cp);\n", arg->name, arg->name);
            fprintf(outFile, "\t\t\t\t if (res != 0){\n");
            fprintf(outFile, "\t\t\t\t\t cerr << \"[C Binding] Error storing object %s as \"<< %s_dest_id_str <<\" in cache \" << endl << flush;\n", arg->name, arg->name);
            fprintf(outFile, "\t\t\t\t\t return res;\n");
            fprintf(outFile, "\t\t\t\t }\n");
            fprintf(outFile, "\t\t\t\t if(is_debug()) cout << \"[C Binding] Object \" << %s_dest_id_str << \" has been added to the cache with type %d.\" << endl << flush;\n", arg->name, position);
            //File not created is out
            /*fprintf(outFile, "\t\t\t\t if (serializeOuts < 1){\n");
            fprintf(outFile, "\t\t\t\t\t remove(%s_dest_id);\n", arg->name);
            fprintf(outFile, "\t\t\t\t\t cout << \"[C Binding] Removed file \" << %s_dest_id << endl;\n", arg->name);
            fprintf(outFile, "\t\t\t\t }\n");*/
        }
        break;
    default:
        ;

    }

    //No data information case (required for GAT Adaptor)
    fprintf(outFile, "\t\t\t } else { //Obtain object as file \n");
    fprintf(outFile, "\t\t\t\t if(is_debug()) cout << \"[C Binding] Data ID for parameter %s not found in argument value. Reseting filename to \" << %s_filename_og << endl << flush;\n", arg->name, arg->name);
    fprintf(outFile, "\t\t\t\t %s_filename = %s_filename_og ;\n", arg->name, arg->name);
    fprintf(outFile, "\t\t\t\t %s_dest_id = %s_filename_og ;\n", arg->name, arg->name);
    if (arg->dir != out_dir){
    	fprintf(outFile, "\t\t\t\t compss_pointer %s_cp;\n", arg->name);
    	fprintf(outFile, "\t\t\t\t %s_cp.type= %d;\n", arg->name, position);
    	fprintf(outFile, "\t\t\t\t %s_cp.elements= %s_elements;\n", arg->name, arg->name);
    	fprintf(outFile, "\t\t\t\t %s_cp.pointer = (void*)%s;\n", arg->name, arg->name);
    	fprintf(outFile, "\t\t\t\t %s_cp.size = %s_cp.elements*sizeof(%s);\n", arg->name, arg->name, arg->classname);
    	fprintf(outFile, "\t\t\t\t int res = cache->deserializeFromFile(%s_filename, %s_cp);\n", arg->name, arg->name);
    	fprintf(outFile, "\t\t\t\t if (res != 0){\n");
    	fprintf(outFile, "\t\t\t\t\t cout << \"[C Binding] Error deserializing object %s from file \"<< %s_filename << endl << flush;\n", arg->name, arg->name);
    	fprintf(outFile, "\t\t\t\t\t return res;\n");
    	fprintf(outFile, "\t\t\t\t }\n");
    	fprintf(outFile, "\t\t\t\t %s = (%s*)%s_cp.pointer;\n", arg->name, arg->classname, arg->name);
    }
    fprintf(outFile, "\t\t\t } //end obtain object as file \n");
    fprintf(outFile, "\t\t\t arg_offset += 1;\n\n");
}

static void add_file_arg_worker_treatment(FILE *outFile, argument *arg) {
    fprintf(outFile, "\t\t\t arg_offset += %d;\n",ARGS_OFFSET);
    fprintf(outFile, "\t\t\t %s_og = strdup(argv[arg_offset]);\n", arg->name);
    fprintf(outFile, "\t\t\t char *%s, *%s_orig_id, *%s_dest_id, *%s_pres_data, *%s_write_data;\n", arg->name, arg->name, arg->name, arg->name, arg->name);
    fprintf(outFile, "\t\t\t %s = %s_og;\n", arg->name, arg->name);
    fprintf(outFile, "\t\t\t %s_orig_id = strsep(&%s,\":\");\n", arg->name, arg->name);
    fprintf(outFile, "\t\t\t if (%s_orig_id != NULL && %s != NULL){ \n", arg->name, arg->name);
    fprintf(outFile, "\t\t\t\t %s_dest_id = strsep(&%s,\":\");\n", arg->name, arg->name);
    fprintf(outFile, "\t\t\t\t %s_pres_data = strsep(&%s,\":\");\n", arg->name, arg->name);
    fprintf(outFile, "\t\t\t\t %s_write_data = strsep(&%s,\":\");\n", arg->name, arg->name);
    fprintf(outFile, "\t\t\t } else {\n");
    fprintf(outFile, "\t\t\t\t %s = %s_og;\n", arg->name, arg->name);
    fprintf(outFile, "\t\t\t }\n");
    fprintf(outFile, "\t\t\t arg_offset += 1;\n\n");
}

static void add_other_arg_worker_treatment(FILE *outFile, argument *arg, Types current_types, bool is_return) {
    // arg_offset -> type
    // arg_offset+1 -> stream
    // arg_offset+2 -> prefix
    // arg_offset+3 -> value
    if (arg->dir == in_dir) {
        fprintf(outFile, "\t\t\t arg_offset += %d;\n", ARGS_OFFSET);
        switch (arg->type) {
        case char_dt:
        case wchar_dt:
            fprintf(outFile, "\t\t\t %s = argv[arg_offset][0];\n", arg->name);
            break;
        case boolean_dt:
            fprintf(outFile, "\t\t\t %s = argv[arg_offset]? 1 : 0;\n", arg->name);
            break;
        case enum_dt:
            fprintf(outFile, "\t\t\t %s = static_cast<%s>(atoi(argv[arg_offset]));\n", arg->name, arg->classname);
            break;
        case int_dt:
        case short_dt:
            fprintf(outFile, "\t\t\t %s = atoi(argv[arg_offset]);\n", arg->name);
            break;
        case long_dt:
            fprintf(outFile, "\t\t\t %s = atol(argv[arg_offset]);\n", arg->name);
            break;
        case longlong_dt:
            fprintf(outFile, "\t\t\t %s = atoll(argv[arg_offset]);\n", arg->name);
            break;
        case float_dt:
            fprintf(outFile, "\t\t\t istringstream %sss(argv[arg_offset]);\n", arg->name);
            fprintf(outFile, "\t\t\t if (!(%sss >> %s)) { \n", arg->name, arg->name);
            fprintf(outFile, "\t\t\t\t cout << \"[C Binding] Error converting %s, string to float\" << endl << flush;", arg->name);
            fprintf(outFile, "\t\t\t \n}");
            break;
        case double_dt:
            fprintf(outFile, "\t\t\t istringstream %sss(argv[arg_offset]);\n", arg->name);
            fprintf(outFile, "\t\t\t if (!(%sss >> %s)) { \n", arg->name, arg->name);
            fprintf(outFile, "\t\t\t\t cout << \"[C Binding] Error converting %s, string to double\" << endl << flush;", arg->name);
            fprintf(outFile, "\t\t\t \n}");
            break;
        case string_dt:
        case string_64_dt:
        case wstring_dt:
            fprintf(outFile, "\t\t\t int %s_nwords = atoi(argv[arg_offset]);\n", arg->name);
            fprintf(outFile, "\t\t\t int word_i_%s;\n", arg->name);
            fprintf(outFile, "\t\t\t int %s_size = 0;\n", arg->name);
            fprintf(outFile, "\t\t\t for (word_i_%s=1; word_i_%s<=%s_nwords; word_i_%s++) {\n", arg->name, arg->name, arg->name, arg->name);
            fprintf(outFile, "\t\t\t\t %s_size += strlen(argv[arg_offset + word_i_%s]);\n", arg->name, arg->name);
            fprintf(outFile, "\t\t\t }\n");
            fprintf(outFile, "\t\t\t %s = (char *) malloc(%s_size + %s_nwords);\n", arg->name,arg->name,arg->name);
            fprintf(outFile, "\t\t\t for (word_i_%s=1; word_i_%s<=%s_nwords; word_i_%s++) {\n", arg->name, arg->name, arg->name, arg->name);
            fprintf(outFile, "\t\t\t\t arg_offset += 1;\n");
            fprintf(outFile, "\t\t\t\t if (word_i_%s == 1)\n", arg->name);
            fprintf(outFile, "\t\t\t\t\t strcpy(%s, argv[arg_offset]);\n", arg->name);
            fprintf(outFile, "\t\t\t\t else {\n");
            fprintf(outFile, "\t\t\t\t\t strcat(%s, \" \");\n", arg->name);
            fprintf(outFile, "\t\t\t\t\t strcat(%s, argv[arg_offset]);\n", arg->name);
            fprintf(outFile, "\t\t\t\t }\n");
            fprintf(outFile, "\t\t\t }\n");
            break;
        default:
            ;
        }
        fprintf(outFile, "\t\t\t arg_offset += 1;\n\n");
    } else {
        add_object_or_array_arg_worker_treatment(outFile, arg, ARGS_OFFSET, current_types, is_return);
    }
}

static void treat_worker_argument(FILE *outFile, argument *arg, Types current_types, bool is_return) {
    char* enum_name;
    int printed_chars = 0;
    switch (arg->type) {
    case enum_dt:

        printed_chars = asprintf(&enum_name, "%s %s", c_out_types[arg->type], arg->classname);
         
        if (printed_chars < 0) {
            printf("ERROR: Not possible to generate enum name and type.\n");
            free(enum_name);
            exit(1);
        }

        fprintf(outFile, "\t\t\t %s %s;\n", enum_name, arg->name);
        add_other_arg_worker_treatment(outFile, arg, current_types, is_return);
        break;
    case char_dt:
    case wchar_dt:
    case boolean_dt:
    case short_dt:
    case long_dt:
    case longlong_dt:
    case int_dt:
    case float_dt:
    case double_dt:
    case string_dt:
    case string_64_dt:
    case wstring_dt:
        fprintf(outFile, "\t\t\t %s %s;\n", c_out_types[arg->type], arg->name);
        add_other_arg_worker_treatment(outFile, arg, current_types, is_return);
        break;
    case object_dt:
        fprintf(outFile, "\t\t\t %s* %s;\n", arg->classname, arg->name);
        add_object_or_array_arg_worker_treatment(outFile, arg, ARGS_OFFSET, current_types, is_return);
        break;
    case array_char_dt:
    case array_byte_dt:
    case array_short_dt:
    case array_int_dt:
    case array_long_dt:
    case array_float_dt:
    case array_double_dt:
        fprintf(outFile, "\t\t\t %s* %s;\n", c_out_types[arg->type], arg->name);
        add_object_or_array_arg_worker_treatment(outFile, arg, ARGS_OFFSET, current_types, is_return);
        break;
    case file_dt:
        fprintf(outFile, "\t\t\t char *%s_og;\n", arg->name);
        add_file_arg_worker_treatment(outFile, arg);
        break;
    case void_dt:
    case any_dt:
    case null_dt:
    default:
        ;
    }
}

static void add_argument_worker_taskcall(FILE *outFile, argument *arg) {
    if (arg->dir == in_dir) {
        switch (arg->type) {
        case char_dt:
        case wchar_dt:
        case boolean_dt:
        case short_dt:
        case long_dt:
        case longlong_dt:
        case enum_dt:
        case int_dt:
        case float_dt:
        case double_dt:
            fprintf(outFile, "%s", arg->name);
            break;
        case object_dt:
            fprintf(outFile, "%s", arg->name);
            break;
        case string_dt:
        case string_64_dt:
        case wstring_dt:
            fprintf(outFile, "%s", arg->name);
            break;
        case file_dt:
            fprintf(outFile, "%s", arg->name);
            break;
        case array_char_dt:
        case array_byte_dt:
        case array_short_dt:
        case array_int_dt:
        case array_long_dt:
        case array_float_dt:
        case array_double_dt:
            fprintf(outFile, "%s", arg->name);
            break;
        case void_dt:
        case any_dt:
        case null_dt:
        default:
            ;
        }
    } else { /* out_dir || inout_dir */
        switch (arg->type) {
        case char_dt:
        case wchar_dt:
        case boolean_dt:
        case short_dt:
        case long_dt:
        case longlong_dt:
        case enum_dt:
        case int_dt:
        case float_dt:
        case double_dt:
            fprintf(outFile, "&%s", arg->name);
            break;
        case object_dt:
            fprintf(outFile, "%s", arg->name);
            break;
        case string_dt:
        case string_64_dt:
        case wstring_dt:
            fprintf(outFile, "&%s", arg->name);
            break;
        case file_dt:
            fprintf(outFile, "%s", arg->name);
            break;
        case array_char_dt:
        case array_byte_dt:
        case array_short_dt:
        case array_int_dt:
        case array_long_dt:
        case array_float_dt:
        case array_double_dt:
            fprintf(outFile, "%s", arg->name);
            break;
        case void_dt:
        case any_dt:
        case null_dt:
        default:
            ;
        }
    }
}

static void add_argument_serialization_worker(FILE *outFile, argument *arg, char *file_suffix, char *elems_suffix, char *tabs) {
    if (arg->dir == out_dir || arg->dir == inout_dir) {
        switch (arg->type) {
        case char_dt:
        case wchar_dt:
        case boolean_dt:
        case short_dt:
        case long_dt:
        case longlong_dt:
        case enum_dt:
        case int_dt:
        case float_dt:
        case double_dt:
        case object_dt:
        case string_dt:
        case string_64_dt:
        case wstring_dt:
        case array_char_dt:
        case array_byte_dt:
        case array_short_dt:
        case array_int_dt:
        case array_long_dt:
        case array_float_dt:
        case array_double_dt: {
            char *elements = concat(arg->name, elems_suffix);
            char *filename = concat(arg->name, file_suffix);
            add_serialization(outFile, arg->type, arg->name, filename, elements, tabs);
            free(filename);
            free(elements);
            fprintf(outFile, "\n");
            break;
        }
        case file_dt:
        case void_dt:
        case any_dt:
        case null_dt:
        default:
            ;
        }
    }

}

static void add_argument_free(FILE *outFile, argument *arg) {
    switch (arg->type) {
    case char_dt:
    case wchar_dt:
    case boolean_dt:
    case short_dt:
    case long_dt:
    case longlong_dt:
    case enum_dt:
    case int_dt:
    case float_dt:
    case double_dt:
        if (arg->dir != in_dir) {
            fprintf(outFile, "\t\t\t free(%s_filename_og);\n", arg->name);
        }
        break;
    case object_dt:
        fprintf(outFile, "\t\t\t free(%s_filename_og);\n", arg->name);
        break;
    case file_dt:
        fprintf(outFile, "\t\t\t free(%s_og);\n", arg->name);
        break;
    case string_dt:
    case string_64_dt:
    case wstring_dt:
        if (arg->dir != in_dir) {
            fprintf(outFile, "\t\t\t free(%s_filename_og);\n", arg->name);
        }
        fprintf(outFile, "\t\t\t free(%s);\n", arg->name);
        break;
    case array_char_dt:
    case array_byte_dt:
    case array_short_dt:
    case array_int_dt:
    case array_long_dt:
    case array_float_dt:
    case array_double_dt:
        //TODO: Add array cases
        fprintf(outFile, "\t\t\t\t //Add array free management\n");
        break;
    case void_dt:
    case any_dt:
    case null_dt:
    default:
        break;
    }
}

static void add_return_object_store(FILE *outFile, argument *arg, Types current_types) {
    Type t;
    t.name = arg->classname;
    t.elements = arg->elements;
    t.dt = arg->type;

    int position = getTypeNumber(t, current_types);
    if (position < 0) {
        printf("ERROR: Position for type (%s,%s,%d) not found", arg->classname, arg->elements, arg->type);
        printAllTypes(current_types);
        exit(1);
    }

    fprintf(outFile, "\t\t\t if (%s_dest_id != NULL){\n", arg->name);
    fprintf(outFile, "\t\t\t\t // Object have to be store in cache\n");
    fprintf(outFile, "\t\t\t\t compss_pointer %s_cp;\n", arg->name);
    fprintf(outFile, "\t\t\t\t %s_cp.type= %d;\n", arg->name, position);
    fprintf(outFile, "\t\t\t\t %s_cp.elements= %s_elements;\n", arg->name, arg->name);
    fprintf(outFile, "\t\t\t\t %s_cp.pointer = (void*)%s;\n", arg->name, arg->name);
    fprintf(outFile, "\t\t\t\t %s_cp.size = %s_cp.elements*sizeof(%s);\n", arg->name, arg->name, arg->classname);
    fprintf(outFile, "\t\t\t\t if(is_debug()) cout << \"[C Binding] Storing object \" << %s_dest_id << \" to cache with type %d.\" << endl << flush;\n", arg->name, position);
    fprintf(outFile, "\t\t\t\t cache->storeInCache(%s_dest_id, %s_cp);\n", arg->name, arg->name);
    fprintf(outFile, "\t\t\t\t if(is_debug()) cout << \"[C Binding] Object \" << %s_dest_id << \" has been added to cache with type %d.\" << endl << flush;\n", arg->name, position);
    fprintf(outFile, "\t\t\t } \n");
}

static void generate_worker_case(FILE *outFile, Types current_types, function *func) {
    argument *arg, ret, th;
    int j = 0;
    int is_first_arg = 1;

    char *func_name = strdup(func->name);
    replace_char(func_name, ':', '_');
    fprintf(outFile, "\t case %sOp:\n", func_name);
    fprintf(outFile, "\t\t {\n"); //Open enum case

    printf("\t\t Adding parameter unmarshalling...\n");
    fflush(NULL);

    arg = func->first_argument;
    while (arg != NULL) {
        treat_worker_argument(outFile, arg, current_types, false);
        arg = arg->next_argument;
    }
    fprintf(outFile, "\t\t\t \n");

    //Managing object callee
    if ((func->classname != NULL) && (func->access_static == 0)) {
        th.name="target_obj";
        th.type=object_dt;
        th.dir=inout_dir;
        th.classname=func->classname;
        th.elements = "0";

        printf("\t\t Adding target object unmarshalling...\n");
        fflush(NULL);

        fprintf(outFile, "\t\t\t%s* %s = new %s();\n", th.classname, th.name, th.classname);

        add_object_or_array_arg_worker_treatment(outFile, &th, ARGS_OFFSET, current_types, false);
    }
    fprintf(outFile, "\t\t\t \n");
    
    //Managing return type
    if (func->return_type == int_dt) {
        printf("\t\t WARNING: Return type int_dt is not implemented, but permited.\n");
    }
    else if (func->return_type != void_dt) {
        ret.name="return_obj";
        ret.type=func->return_type;
        ret.dir=out_dir;
        ret.classname=func->return_typename;
        ret.elements=func->return_elements;

        printf("\t\t Adding return object unmarshalling...\n");

        treat_worker_argument(outFile, &ret, current_types, true);
    }

    fprintf(outFile, "\t\t\t \n");

    fprintf(outFile, "#ifndef OMPSS2_ENABLED\n"); //If OMPSS2_ENABLED not defined

    //Add function call

    printf("\t\t Adding function call unmarshalling...\n");
    fflush(NULL);
    fprintf(outFile, "\t\t\t if(is_debug()) cout << \"[C Binding] Calling function %s.%s\" << endl << flush;\n", func->classname, func->methodname);

    int printed_chars = 0;    
    char* func_to_execute;

    /*
     * The next if statements treat the case of :
     *      - Function is member of a class and is not static and the return type is void                                   :       classE->methodA( .....
     *      - Function is member of a class and is not static and the return type is not void                               : K =   classE->methodA( ....
     *      - Function is member of a class and the acces is static or is not member of a class and the return is not void  : K =   methodA( ....
     *      - Function is member of a class and the acces is static or is not member of a class and the return is void      :       methodA( ....
     */
    if (( func->classname != NULL ) && (func->access_static == 0) && (func->return_type == void_dt)) {
        printed_chars = asprintf(&func_to_execute, "\t\t\t %s->%s(", th.name, func->methodname);
    } else if (( func->classname != NULL ) && (func->access_static == 0) && (func->return_type != void_dt)) {
        if (func->return_type == object_dt) {
            printed_chars = asprintf(&func_to_execute, "\t\t\t *%s = %s->%s(", ret.name, th.name, func->methodname);
        } 
        else {
            printed_chars = asprintf(&func_to_execute, "\t\t\t %s = %s->%s(", ret.name, th.name, func->methodname);
        }
        /*else if (func->return_type == int_dt) {
            printed_chars = asprintf(&func_to_execute, "\t\t\t %s->%s(", th.name, func->methodname);
        }*/
    }
    else if (func->return_type == int_dt) {
        printed_chars = asprintf(&func_to_execute, "\t\t\t %s(", func->name);
    }
    else if (func->return_type != void_dt) {
        printed_chars = asprintf(&func_to_execute, "\t\t\t %s = %s(", ret.name, func->name);
    } 
    else {
        printed_chars = asprintf(&func_to_execute, "\t\t\t %s(", func->name);
    }

    if (printed_chars < 0) {
        printf("ERROR: Not possible to generate method execution.\n");
        free(func_to_execute);
        exit(1);
    }

    fprintf(outFile, "%s", func_to_execute);

    is_first_arg = 1;
    arg = func->first_argument;
    while (arg != NULL) {
        if (is_first_arg) {
            is_first_arg = 0;
        } else {
            fprintf(outFile, ", ");
        }
        add_argument_worker_taskcall(outFile, arg);
        arg = arg->next_argument;
    } 
    fprintf(outFile, ");\n");

    fprintf(outFile, "\t\t\t if(is_debug()) cout << \"[C Binding] Execution of function %s.%s finished.\" << endl << flush;\n", func->classname, func->methodname);
    fprintf(outFile, "\n");

    fprintf(outFile, "#else\n"); //Else spawn OmpSs-2 function with Nanos6


    //Declare the struct holding the arguments of the function
    fprintf(outFile, "\t\t\t%s_struct_t %s_struct;\n", func->methodname, func->methodname);

    assign_arguments_to_struct(outFile, func); //Assign parameters to the struct

    fprintf(outFile, "\t\t\t \n");

    //Add function call
    printf("\t\t Adding function call unmarshalling...\n");
    fflush(NULL);
    fprintf(outFile, "\t\t\t if(is_debug()) cout << \"[C Binding] Spawning function %s.%s as OmpSs-2 task\" << endl << flush;\n", func->classname, func->methodname);

    fprintf(outFile, "\t\t\tcondition_variable_t cond_var = {PTHREAD_MUTEX_INITIALIZER, PTHREAD_COND_INITIALIZER, 0};\n");

    char* nanos6_spawner;
    printed_chars = asprintf(&nanos6_spawner, "\t\t\tnanos6_spawn_function(%s_wrapper, &%s_struct, %s, %s, \"%s_spawned_task\");\n",
            func->methodname, func->methodname, "condition_variable_callback", "&cond_var", func->methodname);

    if (printed_chars < 0) {
        asprintf_error(nanos6_spawner, "Not possible to generate function spawn.\n");
    }

    fprintf(outFile, "%s\n", nanos6_spawner);

    fprintf(outFile, "\t\t\twait_condition_variable(&cond_var);\n");

    if (func->return_type == int_dt && func->return_type != NULL) {
        printf("\t\t WARNING: Return type int_dt is not implemented, but permited.\n");
    }
    else if (func->return_type != void_dt && func->return_type != NULL) {
        char* return_var;
        printed_chars = asprintf(&return_var, "return_obj = %s_struct.ret;\n", func->methodname);

        fprintf(outFile, "%s", return_var);

        if (printed_chars < 0) {
            asprintf_error(return_var, "Not possible to generate return variable.\n");
        }
    }

    fprintf(outFile, "#endif\n");

    //Add return_type store in cache
    if (func->return_type == int_dt) {
        printf("\t\t WARNING: Return type int_dt is not implemented, but permited.\n");
    }
    else if (func->return_type != void_dt) {
        add_return_object_store(outFile, &ret, current_types);
    }
    printf("\t\t Adding out parameter marshalling...\n");
    fflush(NULL);

    //Add output serialization if required
    fprintf(outFile, "\t\t\t if (serializeOuts){\n");

    fprintf(outFile, "\t\t\t\t if(is_debug()) cout << \"[C Binding] Object will be serialized as output.\" << endl << flush;\n");
    if (( func->classname != NULL ) && (func->access_static == 0)) {
        fprintf(outFile, "\t\t\t\t if(is_debug()) cout << \"[C Binding] Treating target object\" << endl << flush;\n");
        add_argument_serialization_worker(outFile, &th, "_dest_id", "_elements", "\t\t\t\t");
        j++;
    }

    if (func->return_type != void_dt) {
        fprintf(outFile, "\t\t\t\t if(is_debug()) cout << \"[C Binding] Treating return object\" << endl << flush;\n");
        add_argument_serialization_worker(outFile, &ret, "_dest_id", "_elements", "\t\t\t\t");
        j++;
    }

    is_first_arg = 1;
    arg = func->first_argument;
    while (arg != NULL) {
        add_argument_serialization_worker(outFile, arg, "_dest_id", "_elements", "\t\t\t\t");
        arg = arg->next_argument;
        j++;
    }

    fprintf(outFile, "\t\t\t if(is_debug()) cout << \"[C Binding] Object has been serialized as output.\" << endl << flush;\n");

    fprintf(outFile, "\t\t\t }\n"); //end of if serialization clause

    printf("\t\t Adding memory free...\n");
    fflush(NULL);

    if (( func->classname != NULL ) && (func->access_static == 0)) {
        add_argument_free(outFile, &th);
    }

    if ( func->return_type != void_dt ) {
        add_argument_free(outFile, &ret);
    }
    arg = func->first_argument;
    while (arg != NULL) {
        add_argument_free(outFile, arg);
        arg = arg->next_argument;
    }

    fprintf(workerFile, "\t\t if (is_debug()){\n");
    fprintf(workerFile, "\t\t\t cout << \"[C-BINDING] Data in cache after execution\" << endl;\n");
    fprintf(workerFile, "\t\t\t cache->printValues();\n");
    fprintf(workerFile, "\t }\n");

    // Operation correctly executed
    fprintf(outFile, "\t\t\t return 0;\n");

    // Close enum case
    fprintf(outFile, "\t\t }\n");
    fprintf(outFile, "\t\t break;\n");

}

/*
 * Generates a struct containing the return variable
 */
static void generate_struct_nanos6_wrapper(FILE *outFile, Types current_types, function *func) {
    argument *arg;
    fprintf(outFile, "typedef struct {\n", func->methodname);

    char* return_type = construct_returntype(func);

    if (func->return_type != void_dt && return_type != NULL) {
        fprintf(outFile, "\t%s ret;\n", return_type);
    }

    arg = func->first_argument;
    while (arg != NULL) {
        fprintf(outFile, "\t%s;\n", construct_type_and_name(arg));
        arg = arg->next_argument;
    }
    fprintf(outFile, "} %s_struct_t;\n", func->methodname);
}

static void generate_function_structs_nanos6(FILE *outFile, Types current_types, function *current_function) {
    /* Nanos6 structs in the application include file */
    while(current_function != NULL) {

        generate_struct_nanos6_wrapper(outFile, current_types, current_function);

        current_function = current_function->next_function;
    }
}

void generate_worker_executor(Types current_types) {

    fprintf(workerFile, "#ifdef OMPSS2_ENABLED\n");

    generate_cond_var_struct(workerFile);

    generate_function_structs_nanos6(workerFile, current_types, get_first_function());

    generate_cond_var_auxiliar_functions(workerFile);

    fprintf(workerFile, "\n");

    function *current_function;
    current_function = get_first_function();

    while (current_function != NULL) {
        generate_nanos6_wrapper(workerFile, current_function);
        current_function = current_function->next_function;
    }
    fprintf(workerFile, "#endif\n\n");

    generate_executor_prototype(workerFile);
    // Args consistent with Runtime [0, NUM_INTERNAL_ARGS]: executable, tracing, taskId, workerDebug, storageConf, method_type, className, methodName,
    //                                                      numSlaves, [slaves], numCus, hasTarget, returnType, numAppParams
    fprintf(workerFile, "\t if(is_debug()){\n"); //Open if debug
    fprintf(workerFile, "\t\t cout << endl;\n");
    fprintf(workerFile, "\t\t cout << \"[C-BINDING] ----------------- C WORKER -----------------\" << endl;\n");
    fprintf(workerFile, "\t\t cout << \"[C-BINDING] Total number of parameters: \" << argc << endl;\n");
    fprintf(workerFile, "\t }\n"); //Close if debug

    fprintf(workerFile, "\t if (argc < MIN_NUM_INTERNAL_ARGS) {\n"); //Open args check if
    fprintf(workerFile, "\t\t cout << \"[C-BINDING] ERROR: Incorrect number of COMPSs internal parameters\"<< endl;\n");
    fprintf(workerFile, "\t\t cout << \"[C-BINDING] Aborting...\" << endl;\n");
    fprintf(workerFile, "\t\t return -1; \n");
    fprintf(workerFile, "\t }\n"); //Close args check if
    fprintf(workerFile, "\n");

    // Log args
    fprintf(workerFile, "\t if(is_debug()){\n"); //Open if debug clause
    fprintf(workerFile, "\t\t cout << \"[C-BINDING] Executable: \" << argv[0] << endl;\n");
    fprintf(workerFile, "\t\t cout << \"[C-BINDING] Tracing: \" <<  argv[1] << endl;\n");
    fprintf(workerFile, "\t\t cout << \"[C-BINDING] Task Id: \" << argv[2] << endl;\n");
    fprintf(workerFile, "\t\t cout << \"[C-BINDING] Worker Debug: \" << argv[3] << endl;\n");
    fprintf(workerFile, "\t\t cout << \"[C-BINDING] StorageConf: \" << argv[4] << endl;\n");
    fprintf(workerFile, "\t\t cout << \"[C-BINDING] MethodType: \" << argv[5] << endl;\n");
    fprintf(workerFile, "\t\t cout << \"[C-BINDING] ClassName: \" << argv[6] << endl;\n");
    fprintf(workerFile, "\t\t cout << \"[C-BINDING] MethodName: \" << argv[7] << endl;\n");
    fprintf(workerFile, "\t\t cout << \"[C-BINDING] TimeOut: \" << argv[8] << endl;\n");
    fprintf(workerFile, "\t\t cout << \"[C-BINDING] NumSlaves: \" << argv[9] << endl;\n");
    fprintf(workerFile, "\t }\n"); //Close if debug

    fprintf(workerFile, "\t int numSlaves=atoi(argv[9]);\n");

    fprintf(workerFile, "\t if(is_debug()){\n"); //Open if debug clause
    fprintf(workerFile, "\t\t for (int i = 0; i < numSlaves; ++i) {\n"); //Open for
    fprintf(workerFile, "\t\t\t cout <<\"[C-BINDING] Slave \" << i << \" has name \" << argv[NUM_BASE_ARGS + i] << endl;\n");
    fprintf(workerFile, "\t\t }\n"); //Close for
    fprintf(workerFile, "\t }\n"); //Close if debug

    fprintf(workerFile, "\t int NUM_INTERNAL_ARGS=NUM_BASE_ARGS + numSlaves;\n");

    //fprintf(workerFile, "\t if(is_debug()){\n"); //Open if debug clause
    fprintf(workerFile, "\t\t cout << \"[C-BINDING] NumComputingUnits: \" << argv[NUM_INTERNAL_ARGS++] << endl;\n");
    fprintf(workerFile, "\t\t cout << \"[C-BINDING] HasTarget: \" << argv[NUM_INTERNAL_ARGS++] << endl;\n");
    fprintf(workerFile, "\t\t cout << \"[C-BINDING] NumReturns: \" << argv[NUM_INTERNAL_ARGS++] << endl;\n");
    fprintf(workerFile, "\t\t cout << \"[C-BINDING] ReturnType: \" << argv[NUM_INTERNAL_ARGS++] << endl;\n");
    fprintf(workerFile, "\t\t cout << \"[C-BINDING] Num App Params: \" << argv[NUM_INTERNAL_ARGS++] << endl;\n");
    fprintf(workerFile, "\t\t cout << \"[C-BINDING] Application Arguments:\" << endl;\n");
    fprintf(workerFile, "\t\t for(int i = NUM_INTERNAL_ARGS; i < argc; i++) { \n"); //Open for
    fprintf(workerFile, "\t\t\t cout << \"\t\" << argv[i] << endl;\n");
    fprintf(workerFile, "\t\t }\n"); //Close for clause
    fprintf(workerFile, "\t\t cout << flush;\n");
    fprintf(workerFile, "\n");
    //fprintf(workerFile, "\t }\n"); //Close if debug clause

    // Get OpName and OpCode
    fprintf(workerFile, "\t enum operationCode opCod;\n");
    fprintf(workerFile, "\t char *opName = strdup(argv[METHOD_NAME_POS]);\n");
    fprintf(workerFile, "\t for(int i=0; i < N_OPS; i++) {\n");
    fprintf(workerFile, "\t\t if(strcmp(operationName[i], opName) == 0) {\n");
    fprintf(workerFile, "\t\t\t opCod=(enum operationCode)i;\n");
    fprintf(workerFile, "\t\t\t break;\n");
    fprintf(workerFile, "\t\t }\n");
    fprintf(workerFile, "\t }\n");

    fprintf(workerFile, "\t if(is_debug()){\n"); // Open if debug clause
    fprintf(workerFile, "\t\t cout << \"[C-BINDING] OpName: \" << opName << \"OpCode: \" << (int)opCod << endl;\n");
    fprintf(workerFile, "\t\t cout << \"[C-BINDING] --------------------------------------------\"<< endl << endl << flush;\n");
    fprintf(workerFile, "\t\t cout << \"[C-BINDING] Data in cache before executing:\" << endl;\n");
    fprintf(workerFile, "\t\t cache->printValues();\n");
    fprintf(workerFile, "\t }\n");//Close if debug clause

    // OpCode switch
    fprintf(workerFile, "\t int arg_offset = NUM_INTERNAL_ARGS;\n");
    fprintf(workerFile, "\t switch(opCod)\n");
    fprintf(workerFile, "\t {\n"); //Open switch clause
}

void generate_body(void) {
    Types current_types;
    initTypes(&current_types);
    function *current_function;
    generate_worker_executor(current_types);

    //Check if there is any include in the .idl file
    include* inc = get_first_include();
    add_include_headers(inc);

    generate_enum(includeFile, get_first_function());

    fprintf(stubsFile, "\n");

    fprintf(includeFile, "/* Functions to be implemented. We suggest that you create a file */\n");
    fprintf(includeFile, "/* with name '%s-functions.cc' and implement them there. */\n", get_filename_base());

    current_function = get_first_function();

    while (current_function != NULL) {

        printf("Treating function: %s\n", current_function->name);

        printf("\t Generating class includes and types... \n");
        fflush(NULL);
        generate_class_includes_and_check_types(includeFile, &current_types, current_function);

        // If the current function is not an object method
        if (strstr(current_function->name, "::") == NULL) {
            printf("\t Generating prototypes in includes files ...\n");
            fflush(NULL);
            generate_prototype(includeFile, current_function);
            fprintf(includeFile, ";\n");
        }

        printf("\t Generating prototype in stubs... \n");
        fflush(NULL);
        generate_prototype(stubsFile, current_function);
        fprintf(stubsFile, "\n");
        fprintf(stubsFile, "{\n");
        printf("\t Generating parameters' buffers in stubs... \n");
        fflush(NULL);
        generate_parameter_buffers(stubsFile, current_function);
        printf("\t Generating parameters' marshalling in stubs... \n");
        fflush(NULL);
        generate_parameter_marshalling(stubsFile, current_function, current_types);
        printf("\t Generating runtime execute calls in stubs... \n");
        fflush(NULL);
        generate_execute_task_call(stubsFile, current_function);
        fprintf(stubsFile, "}\n");
        fprintf(stubsFile, "\n");

        printf("\t Generating worker functions... \n");
        fflush(NULL);
        generate_worker_case(workerFile, current_types, current_function);

        current_function = current_function->next_function;
    }

    current_function = get_first_function();

    generate_executor_end();
    printf("\t Generating remove and serialize functions... \n");
    fflush(NULL);
    generate_initThread_empty(stubsFile);
    generate_finishThread_empty(stubsFile);
    generate_execute_empty(stubsFile);
    generate_remove_func(workerFile, current_types);
    generate_remove_func(stubsFile, current_types);
    generate_copy_func(workerFile, current_types);
    generate_copy_func(stubsFile, current_types);
    generate_serializeToStream_func(workerFile, current_types);
    generate_serializeToStream_func(stubsFile, current_types);
    generate_serializeToFile_func(workerFile, current_types);
    generate_serializeToFile_func(stubsFile, current_types);
    generate_deserializeFromStream_func(workerFile, current_types);
    generate_deserializeFromStream_func(stubsFile, current_types);
    generate_deserializeFromFile_func(workerFile, current_types);
    generate_deserializeFromFile_func(stubsFile, current_types);
    generate_worker_initThread();
    generate_worker_finishThread();
}
